
#ifndef PBBCELIB_H
#error This must be included from pbbcelib.h
#endif

#ifndef __HOSTSIM__
static inline unsigned char __input_port(unsigned char port)
{
    unsigned char result;
    asm volatile (
        "input %0, %1"          // %0 = result, %1 = port
        : /* outputs */
            /*0*/ "=r" (result)
        : /* inputs */
            /*1*/ "ir" (port)
        : /* clobbers */
    );
    return result;
}

static inline void __output_port(unsigned char value, unsigned char port)
{
    asm volatile (
        "output %0, %1"         // %0 = value, %1 = port
        : /* outputs */
        : /* inputs */
            /*0*/ "r" (value),
            /*1*/ "ir" (port)
        : /* clobbers */
    );
}
#else
/** Host Simulation */
unsigned char __input_port(unsigned char port);
void __output_port(unsigned char value, unsigned char port);
#endif

/* The following is from platform/bce_controller.vhd:
-- PicoBlaze I/O Map:
-- 
-- Overview (see details below):
--      0x00 - 0x0F     BCE CFG Memory interface
--      0x10 - 0x1F     HW configuration constants
--      0x20 - 0x2F     DFU control/status
--      0x30 - 0x3F     AG #0
--      0x40 - 0x4F     AG #1
--      0x50 - 0x5F     AG #2
--      0x60 - 0x6F     AG #3
--      ....
-- 
-- NOTE: WO = write-only, RO=read-only, RW=read-write
-- 
-- BCE-Control Memory interface (0x00 - 0x0F)
--      0x00            RO      BCE CFG Control word (@ CFG[0x00]), ll byte
--      0x01            RO      BCE CFG Control word (@ CFG[0x00]), lh byte
--      0x02            RO      BCE CFG Control word (@ CFG[0x00]), hl byte
--      0x03            RO      BCE CFG Control word (@ CFG[0x00]), hh byte
--      0x04            RW      BCE CFG Status word (@ CFG[0x80]), ll byte
--      0x05            RW      BCE CFG Status word (@ CFG[0x80]), lh byte
--      0x06            RW      BCE CFG Status word (@ CFG[0x80]), hl byte
--      0x07            RW      BCE CFG Status word (@ CFG[0x80]), hh byte
--      0x08            WO      'Output' address here to sample from cfg mem into the data regs below
--      0x09            WO      'Output' address here to push from data regs below into the cfg mem
--      0x0A - 0x0B     -
--      0x0C            RW      cfg mem data reg 0 [lsb]
--      0x0D            RW      cfg mem data reg 1
--      0x0E            RW      cfg mem data reg 2
--      0x0F            RW      cfg mem data reg 3 [msb]
-- 
-- HW introspection (0x10 - 0x1F)
--      0x10            RO      DFU Kind
--      0x11            RO      Number of data memories
--      0x12            RO      Number of address generators
--      0x13 - 0x17     -       (reserved)
--      0x18 - 0x1B     RW      Program running time, 32b performance counter; a write clears the counter.
--      0x1C - 0x1D     RO      Running time of the last DFU operation, 16b performance counter.
--      0x1E - 0x1F     -       (reserved)
-- 
-- DFU control/status regs (0x20 - 0x2F)
--      0x20            WO      DFU opcode; the write triggers DFU execution
--      0x21            WO      vector length (count) [lsb]
--      0x22            WO      vector length (count) [msb]
--      0x23            WO      number of repetitions the DFU operation should be performed
--      0x24            WO      EN_VPMEM: Fetch and start instruction from the VP-memory
--      0x24 - 0x2D     -       (reserved)
--      0x2E            RO      bit_N = AG #N is waiting for a memory lock
--      0x2F            RW      bit_0 = Vector operation completed, writing Zero hard-clears the DFU;
--                              bit_1 = DFU_Err; 
--                              bit_2 = (obsolete) Always read Zero.
-- 
-- AG #N interface (0x{N+3}0 - 0x{N+3}F)
--      0xX0            WO      memory selection
--      0xX1            WO      initial address [lsb]
--      0xX2            WO      initial address [msb]
--      0xX3            WO      increment, signed [lsb]
--      0xX4            WO      increment, signed [msb]
--      0xX5            WO      low bound [lsb]
--      0xX6            WO      low bound [msb]
--      0xX7            WO      high bound [lsb]
--      0xX8            WO      high bound [msb]
--      0xX9            WO      AG flags:
--                                      bit_0 = use index AG
--                                      bit_1 = step when the indexing AG is reaching boundary
--      0xXA            WO      EN_VPMEM: Load from VP-MEM
--      0xXB - 0xXF     -       (reserved)
*/

/** the REG_* constants describe the I/O ports for PicoBlaze INPUT/OUTPUT instructions */
/** NOTE: LH = byte 0 (LSB), LH = byte 1, HL = byte 2, HH = byte 3 (MSB) */
/* CFG Memory interface registers */
#define REG_BCECTRL_LL          0x00
#define REG_BCECTRL_LH          0x01
#define REG_BCECTRL_HL          0x02
#define REG_BCECTRL_HH          0x03
//
#define REG_BCESTAT_LL          0x04
#define REG_BCESTAT_LH          0x05
#define REG_BCESTAT_HL          0x06
#define REG_BCESTAT_HH          0x07
//
#define REG_READCFG_AD          0x08
#define REG_WRITECFG_AD         0x09
#define REG_CFGDT_0             0x0C
#define REG_CFGDT_1             0x0D
#define REG_CFGDT_2             0x0E
#define REG_CFGDT_3             0x0F

/* HW config introspection registers */
#define REG_HW_DFU_KIND         0x10
#define REG_HW_NUM_DTMEMS       0x11
#define REG_HW_NUM_AGS          0x12
#define REG_HW_CAPS             0x13

#define REG_PCNT_PRGTIME_LL     0x18
#define REG_PCNT_PRGTIME_LH     0x19
#define REG_PCNT_PRGTIME_HL     0x1A
#define REG_PCNT_PRGTIME_HH     0x1B
#define REG_PCNT_DFUTIME_L      0x1C
#define REG_PCNT_DFUTIME_H      0x1D

/* DFU interface registers */
#define REG_DFUOPC              0x20
#define REG_DFUVECL_L           0x21
#define REG_DFUVECL_H           0x22
#define REG_DFUREPETITIONS      0x23
#define REG_DFUINSN             0x24
#define REG_AGSTAT              0x2E
#define REG_DFUSTAT             0x2F

/* AG interface registers */
#define REG_AG_BASE             0x30
#define OFSREG_AG_MSEL          0x00
#define OFSREG_AG_ADDR_L        0x01
#define OFSREG_AG_ADDR_H        0x02
#define OFSREG_AG_STEP_L        0x03
#define OFSREG_AG_STEP_H        0x04
#define OFSREG_AG_LOBND_L       0x05
#define OFSREG_AG_LOBND_H       0x06
#define OFSREG_AG_HIBND_L       0x07
#define OFSREG_AG_HIBND_H       0x08
#define OFSREG_AG_FLAGS         0x09
#define OFSREG_AG_LDMVP         0x0A

/* RedHorn Network iface */
#define REG_RHCS_FLAGS          0xC0
#define REG_RHCS_TARGET         0xC1
#define REG_RHCS_ENDPOINT       0xC2
#define REG_RHCS_RECVFLKIND     0xC3            /* a write here will pop the input buffer */
#define REG_RHCS_RECVPOP        REG_RHCS_RECVFLKIND
#define REG_RHCS_SNDFLKIND      0xC4
#define REG_RHCS_DATA_LL        0xC8
#define REG_RHCS_DATA_LH        0xC9
#define REG_RHCS_DATA_HL        0xCA
#define REG_RHCS_DATA_HH        0xCB

/* flags in REG_RHCS_FLAGS */
#define RHFL_RECV_PRESENT       0x01
#define RHFL_SND_ALLOWED        0x02

/* flit-kinds in REG_RHCS_RECVFLKIND and REG_RHCS_SNDFLKIND */
#define RHFLK_RMT_READ          0x00            /* remote control/debug read */
#define RHFLK_RMT_WRITE         0x01            /* remote control/debug write */
#define RHFLK_RMT_READRESP      0x02            /* remote control/debug read response */

/* NPI DMA Configuration interface (0xD0 - 0xDF) */
#define REG_DMA_CFGSPACE        0xD0            /* 0xD0 through 0xD7 */
#define REG_DMA_WINDOW          0xDE

#define REG_DMA_EXTADR_LL       0xD0
#define REG_DMA_EXTADR_LH       0xD1
#define REG_DMA_EXTADR_HL       0xD2
#define REG_DMA_EXTADR_HH       0xD3
#define REG_DMA_LOCADR_L        0xD4
#define REG_DMA_LOCADR_H        0xD5
#define REG_DMA_LENGTH_L        0xD6
#define REG_DMA_LENGTH_H        0xD7

// these special registers are in the DMA configuration memory
#define DMAREG_CMDS             0x80
#define DMAREG_STATUS           0xC0

/** the CFGAD_* constants describe the layout of the BCE configuration memory */
#define CFGAD_CTRL              0x00
#define CFGAD_STATUS            0x80

/** bit masks */
// TODO: rename these
/* TAG = two bits (5 downto 4) in both the CFG Control and Status registers */
#define MASK_TAG                0x30            /* in ctrl and status cfg */
#define N_MASK_TAG              0xCF            /* inverted MASK_TAG */
/* Bit 'Go' (bit#0) in CFG Control register; Hardware-controlled. */
#define MASK_G                  0x01
/* Bit 'Running' (bit#0) in CFG Status register */
#define MASK_R                  0x01
/* Bit 'Busy' (bit#1) in CFG Status register */
#define MASK_B                  0x02
#define N_MASK_B                0xFD
#define MASK_BR                 0x03            /* (MASK_B|MASK_R) */
/* Bit 'DFU Done' (bit#0) in DFU Status reg. */
#define MASK_OperationDone      0x01
#define MASK_LicenseOut         0x08


/** inline functions */

static inline void pb2dfu_start_op(unsigned char op, unsigned int cnt)
{
    __output_port(cnt & 0xff,   REG_DFUVECL_L);
    __output_port(cnt >> 8,     REG_DFUVECL_H);
    __output_port(op,           REG_DFUOPC);
}

static inline void pb2dfu_restart_op(unsigned char op)
{
    __output_port(op,   REG_DFUOPC);
}

static inline void pb2dfu_start_insn(unsigned int insn)
{
    __output_port(insn & 0xFF,  REG_DFUINSN);
}

static inline void pb2dfu_set_cnt(unsigned int cnt)
{
    __output_port(cnt & 0xff,   REG_DFUVECL_L);
    __output_port(cnt >> 8,     REG_DFUVECL_H);
}

static inline void pb2dfu_set_inc(unsigned char mem, int inc)
{
    __output_port(inc & 0xff,   mem + OFSREG_AG_STEP_L);
    __output_port(inc >> 8,     mem + OFSREG_AG_STEP_H);
}

static inline void pb2dfu_set_bank(unsigned char mem, unsigned char bank)
{
    __output_port(bank,         mem + OFSREG_AG_MSEL);
}

static inline void pb2dfu_set_addr(unsigned char mem, unsigned int addr)
{
    __output_port(addr & 0xff,          mem + OFSREG_AG_ADDR_L);
    __output_port(addr >> 8,            mem + OFSREG_AG_ADDR_H);
}

static inline void pb2dfu_set_fulladdr(unsigned char mem, unsigned char bank, unsigned int addr)
{
    __output_port(bank,         mem + OFSREG_AG_MSEL);
    __output_port(addr & 0xff,  mem + OFSREG_AG_ADDR_L);
    __output_port((addr >> 8),  mem + OFSREG_AG_ADDR_H);
}

static inline void pb2dfu_set_vector(unsigned char dfuag, unsigned char mvp)
{
    __output_port(mvp,          dfuag + OFSREG_AG_LDMVP);
}

/**
 * pb2dfu_set_agflags - set operation flags/mode of the specified address generator (DFU argument)
 * @dfuag: select the DFU argument number (constant %DFUAG_x)
 * @agflags: bitmap of flags to set (constants %AGFL_x)
 */
static inline void pb2dfu_set_agflags(unsigned char dfuag, unsigned char agflags)
{
    __output_port(agflags,      dfuag + OFSREG_AG_FLAGS);
}

/**
 * pb2dfu_set_repetitions - set the number of repetitions of a DFU operation.
 * @nrep: the number of times the following DFU operation will be restarted.
 */
static inline void pb2dfu_set_repetitions(unsigned char nrep)
{
    __output_port(nrep,         REG_DFUREPETITIONS);
}

/**
 * Sets the R and B bits in the CFG Status register
 * to report that the firmware has sucessfully started and is busy.
 * This function does NOT block.
 * NOTE: This could be integrated into picoblaze C library
 * to be called automatically upon startup.
 */
static inline void pb2mb_report_running()
{
    __output_port(
        __input_port(REG_BCESTAT_LL) | MASK_BR,
        REG_BCESTAT_LL);
}

/**
 * Read the DMA status register.
 * The status bits correspond to the eight DMA channels.
 * A value of 1 in a bit indicates that the corresponding channel is busy,
 * a value of 0 indicates an operation has completed.
 */
static inline unsigned char dma_get_status()
{
    __output_port(DMAREG_STATUS,        REG_DMA_WINDOW);
    return __input_port(REG_DMA_CFGSPACE);
}
