/*******************************************************************************
 * This file is part of the PB2 library developed
 * within the EU Artemis project SMECY (Smart Multicore Embedded Systems)
 * Artemis JU 100230 and MSMT 7H10001,       http://www.smecy.eu
 * Copyright (C) 2011 UTIA AV CR, v.v.i.     http://sp.utia.cz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even 
 * the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE.
 *
 * This file has been released within the SMECY project
 * consortium for the requirements of the SMECY project.
 * Any use outside the SMECY consortium and/or for any
 * developments outside the scope of the SMECY project is prohibited.
 *
 * For more details contact Roman Bartosinski <bartosr@utia.cas.cz>.
 *******************************************************************************
 * Filename  : pbbcelib-impl.h
 * Authors   : Jaroslav Sykora <sykora@utia.cas.cz>
 * Project   : SMECY
 * Purpose   : Header file - API for firmware of BCE hardware accelerator
 * Release   : 
 * Version   : 0.1
 * Date      : 2012/01/18
 *
 * Long Description:
 *  Implementation of inlined functions. Optimized at level 0.
 *  Application should be compiled with pblaze toolchain (based on LLVM) 
 *  
 * 
 *******************************************************************************
 * Modifications:
 *  Author: Roman Bartosinski
 *  Date  : 2012/05/01
 *  Description: Updated comments
 *  --
 *  
 ******************************************************************************/

#ifndef PBBCELIB_H_V3
#error This header file must be included from pbbcelib.h (version 3)
#endif


/* BCE IDs for the WAL_OP_GETID operation */
#define WAL_BCE_JSY_ID_GENERIC_3D  1
#define WAL_BCE_JSY_ID_GENERIC_4D  2


#ifndef __HOSTSIM__
static inline unsigned char __input_port(unsigned char port)
{
    unsigned char result;
    asm volatile (
        "input %0, %1"          // %0 = result, %1 = port
        : /* outputs */
            /*0*/ "=r" (result)
        : /* inputs */
            /*1*/ "ir" (port)
        : /* clobbers */
    );
    return result;
}

static inline void __output_port(unsigned char value, unsigned char port)
{
    asm volatile (
        "output %0, %1"         // %0 = value, %1 = port
        : /* outputs */
        : /* inputs */
            /*0*/ "r" (value),
            /*1*/ "ir" (port)
        : /* clobbers */
    );
}
#else
/** Host Simulation */
unsigned char __input_port(unsigned char port);
void __output_port(unsigned char value, unsigned char port);
#endif

/** the REG_* constants describe the I/O ports for PicoBlaze INPUT/OUTPUT instructions */
/** NOTE: LH = byte 0 (LSB), LH = byte 1, HL = byte 2, HH = byte 3 (MSB) */
/* CFG Memory interface registers */
#define REG_BCECTRL_LL          0x00
#define REG_BCECTRL_LH          0x01
#define REG_BCECTRL_HL          0x02
#define REG_BCECTRL_HH          0x03
//
#define REG_BCESTAT_LL          0x04
#define REG_BCESTAT_LH          0x05
#define REG_BCESTAT_HL          0x06
#define REG_BCESTAT_HH          0x07
//
#define REG_READCFG_AD          0x08
#define REG_WRITECFG_AD         0x09
#define REG_CFGDT_0             0x0C
#define REG_CFGDT_1             0x0D
#define REG_CFGDT_2             0x0E
#define REG_CFGDT_3             0x0F

/* HW config introspection registers */
#define REG_HW_DFU_KIND         0x10
#define REG_HW_NUM_DTMEMS       0x11
#define REG_HW_NUM_AGS          0x12
#define REG_HW_CAPS             0x13

#define REG_PCNT_PRGTIME_LL     0x18
#define REG_PCNT_PRGTIME_LH     0x19
#define REG_PCNT_PRGTIME_HL     0x1A
#define REG_PCNT_PRGTIME_HH     0x1B
#define REG_PCNT_DFUTIME_L      0x1C
#define REG_PCNT_DFUTIME_H      0x1D

/* DFU interface registers */
#define REG_DFUOPC              0x20
#define REG_DFUVECL_L           0x21
#define REG_DFUVECL_H           0x22
#define REG_DFUREPETITIONS      0x23
#define REG_DFUINSN             0x24
#define REG_AGSTAT              0x2E
#define REG_DFUSTAT             0x2F

/* AG interface registers */
#define REG_AG_BASE             0x30
#define OFSREG_AG_MSEL          0x00
#define OFSREG_AG_ADDR_L        0x01
#define OFSREG_AG_ADDR_H        0x02
#define OFSREG_AG_STEP_L        0x03
#define OFSREG_AG_STEP_H        0x04
#define OFSREG_AG_LOBND_L       0x05
#define OFSREG_AG_LOBND_H       0x06
#define OFSREG_AG_HIBND_L       0x07
#define OFSREG_AG_HIBND_H       0x08
#define OFSREG_AG_FLAGS         0x09
#define OFSREG_AG_LDMVP         0x0A

/* RedHorn Network iface */
#define REG_RHCS_FLAGS          0xC0
#define REG_RHCS_TARGET         0xC1
#define REG_RHCS_ENDPOINT       0xC2
#define REG_RHCS_RECVFLKIND     0xC3            /* a write here will pop the input buffer */
#define REG_RHCS_RECVPOP        REG_RHCS_RECVFLKIND
#define REG_RHCS_SNDFLKIND      0xC4
#define REG_RHCS_DATA_LL        0xC8
#define REG_RHCS_DATA_LH        0xC9
#define REG_RHCS_DATA_HL        0xCA
#define REG_RHCS_DATA_HH        0xCB

/* flags in REG_RHCS_FLAGS */
#define RHFL_RECV_PRESENT       0x01
#define RHFL_SND_ALLOWED        0x02

/* flit-kinds in REG_RHCS_RECVFLKIND and REG_RHCS_SNDFLKIND */
#define RHFLK_RMT_READ          0x00            /* remote control/debug read */
#define RHFLK_RMT_WRITE         0x01            /* remote control/debug write */
#define RHFLK_RMT_READRESP      0x02            /* remote control/debug read response */

/* NPI DMA Configuration interface (0xD0 - 0xDF) */
#define REG_DMA_CFGSPACE        0xD0            /* 0xD0 through 0xD7 */
#define REG_DMA_WINDOW          0xDE

#define REG_DMA_EXTADR_LL       0xD0
#define REG_DMA_EXTADR_LH       0xD1
#define REG_DMA_EXTADR_HL       0xD2
#define REG_DMA_EXTADR_HH       0xD3
#define REG_DMA_LOCADR_L        0xD4
#define REG_DMA_LOCADR_H        0xD5
#define REG_DMA_LENGTH_L        0xD6
#define REG_DMA_LENGTH_H        0xD7

// these special registers are in the DMA configuration memory
#define DMAREG_CMDS             0x80
#define DMAREG_STATUS           0xC0

/** the CFGAD_* constants describe the layout of the BCE configuration memory */
#define CFGAD_CTRL              0x00
#define CFGAD_STATUS            0x80

/** bit masks */
// TODO: rename these
/* TAG = two bits (5 downto 4) in both the CFG Control and Status registers */
#define MASK_TAG                0x30            /* in ctrl and status cfg */
#define N_MASK_TAG              0xCF            /* inverted MASK_TAG */
/* Bit 'Go' (bit#0) in CFG Control register; Hardware-controlled. */
#define MASK_G                  0x01
/* Bit 'Running' (bit#0) in CFG Status register */
#define MASK_R                  0x01
/* Bit 'Busy' (bit#1) in CFG Status register */
#define MASK_B                  0x02
#define N_MASK_B                0xFD
#define MASK_BR                 0x03            /* (MASK_B|MASK_R) */
/* Bit 'DFU Done' (bit#0) in DFU Status reg. */
#define MASK_OperationDone      0x01
#define MASK_LicenseOut         0x08


/** inline functions */

static inline void pb2dfu_start_op(unsigned char op, unsigned int cnt)
{
    __output_port(cnt & 0xff,   REG_DFUVECL_L);
    __output_port(cnt >> 8,     REG_DFUVECL_H);
    __output_port(op,           REG_DFUOPC);
}

static inline void pb2dfu_restart_op(unsigned char op)
{
    __output_port(op,   REG_DFUOPC);
}

static inline void pb2dfu_start_insn(unsigned int insn)
{
    __output_port(insn & 0xFF,  REG_DFUINSN);
}

static inline void pb2dfu_set_cnt(unsigned int cnt)
{
    __output_port(cnt & 0xff,   REG_DFUVECL_L);
    __output_port(cnt >> 8,     REG_DFUVECL_H);
}

static inline void pb2dfu_set_inc(unsigned char mem, int inc)
{
    __output_port(inc & 0xff,   mem + OFSREG_AG_STEP_L);
    __output_port(inc >> 8,     mem + OFSREG_AG_STEP_H);
}

static inline void pb2dfu_set_bank(unsigned char mem, unsigned char bank)
{
    __output_port(bank,         mem + OFSREG_AG_MSEL);
}

static inline void pb2dfu_set_addr(unsigned char mem, unsigned int addr)
{
    __output_port(addr & 0xff,          mem + OFSREG_AG_ADDR_L);
    __output_port(addr >> 8,            mem + OFSREG_AG_ADDR_H);
}

static inline void pb2dfu_set_fulladdr(unsigned char mem, unsigned char bank, unsigned int addr)
{
    __output_port(bank,         mem + OFSREG_AG_MSEL);
    __output_port(addr & 0xff,  mem + OFSREG_AG_ADDR_L);
    __output_port((addr >> 8),  mem + OFSREG_AG_ADDR_H);
}

static inline void pb2dfu_set_vector(unsigned char dfuag, unsigned char mvp)
{
    __output_port(mvp,          dfuag + OFSREG_AG_LDMVP);
}

/**
 * pb2dfu_set_agflags - set operation flags/mode of the specified address generator (DFU argument)
 * @dfuag: select the DFU argument number (constant %DFUAG_x)
 * @agflags: bitmap of flags to set (constants %AGFL_x)
 */
static inline void pb2dfu_set_agflags(unsigned char dfuag, unsigned char agflags)
{
    __output_port(agflags,      dfuag + OFSREG_AG_FLAGS);
}

/**
 * pb2dfu_set_repetitions - set the number of repetitions of a DFU operation.
 * @nrep: the number of times the following DFU operation will be restarted.
 */
static inline void pb2dfu_set_repetitions(unsigned char nrep)
{
    __output_port(nrep,         REG_DFUREPETITIONS);
}

/**
 * Sets the R and B bits in the CFG Status register
 * to report that the firmware has sucessfully started and is busy.
 * This function does NOT block.
 * NOTE: This could be integrated into picoblaze C library
 * to be called automatically upon startup.
 * 
 * Return Value: The function does not return any value.
 */
static inline void pb2mb_report_running()
{
    __output_port(
        __input_port(REG_BCESTAT_LL) | MASK_BR,
        REG_BCESTAT_LL);
}

/**
 * dma_get_status - Read the DMA status register.
 * The status bits correspond to the eight DMA channels.
 * A value of 1 in a bit indicates that the corresponding channel is busy,
 * a value of 0 indicates an operation has completed.
 * 
 * Return Value: Bitmap of busy DMA channels.
 */
static inline unsigned char dma_get_status()
{
    __output_port(DMAREG_STATUS,        REG_DMA_WINDOW);
    return __input_port(REG_DMA_CFGSPACE);
}
