----------------------------------------------------------------------------------
-- Detektor objektu
-- Copyright (C) 2012 Brno University of Technology,
--                        Faculty of Information Technology
-- Author(s): Petr Musil <xmusilpetr AT fit.vutbr.cz>
--  
-- LICENSE TERMS
--  
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 	1. Redistributions of source code must retain the above copyright
-- 	notice, this list of conditions and the following disclaimer.
-- 	2. Redistributions in binary form must reproduce the above copyright
-- 	notice, this list of conditions and the following disclaimer in
-- 	the documentation and/or other materials provided with the
-- 	distribution.
-- 	3. All advertising materials mentioning features or use of this software
-- 	or firmware must display the following acknowledgement:
--   
-- 	This product includes software developed by the University of
-- 	Technology, Faculty of Information Technology, Brno and its
-- 	contributors.
--  
-- 	4. Neither the name of the Company nor the names of its contributors
-- 	may be used to endorse or promote products derived from this
-- 	software without specific prior written permission.
--  
-- This software or firmware is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
-- 
--      $Id$
----------------------------------------------------------------------------------


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.std_logic_unsigned.all;
use work.configure.all;		-- my data types
use IEEE.NUMERIC_STD.ALL;
library UNISIM;
use UNISIM.VComponents.all;

entity image_buffer is
	port(
		clk 			: in std_logic;
		reset			: in std_logic;
		
		-- identifikace stavu
		OVERLOAD 	: out std_logic;
		UNDERLOAD	: out std_logic;
		
		-- vstup
		in_data		: in std_logic_vector(31 downto 0);
		in_enable	: in std_logic;
		in_ready		: out std_logic;
		
		-- vystup
		out_data		: out std_logic_vector(71 downto 0);
		out_ready	: out std_logic;
		out_next		: in std_logic
		
	);
end image_buffer;

architecture Behavioral of image_buffer is


-- vstupni automat
type state_type_in is (st0_in, st1_in, st2_in, st3_in, st4_in, st5_in); 
signal state_in, next_state_in : state_type_in; 

signal reorder_mem, reorder_mem_next : std_logic_vector(71 downto 0);
signal mem_w, next_mem_w, last_flag, ready_flag : std_logic;
signal next_in_ready : std_logic;

-- ridici automat
type state_type_co is (st0_co, st1_co, st2_co, st3_co);
signal state_co, next_state_co : state_type_co;

signal in_x, next_in_x, out_x, next_out_x : std_logic_vector (5 downto 0);
signal in_y, next_in_y, out_y, next_out_y : std_logic_vector (2 downto 0);

-- vystupni automat
type state_type_out is (st0_out, st1_out, st2_out, st3_out, st4_out);
signal state_out, next_state_out : state_type_out;

signal next_out_ready : std_logic;
signal mem_addr_w, mem_addr_r : std_logic_vector(8 downto 0);


--------------------------------------------------------------------------------
begin

SYNC_PROC_IN: process (clk)
   begin
      if (clk'event and clk = '1') then
         if (reset = '1') then
            state_in <= st0_in;
            mem_w <= '0';
				reorder_mem <= (others => '0');
				in_ready <= '0';
         else
            state_in <= next_state_in;
            mem_w <= next_mem_w;
				in_ready <= next_in_ready;
				reorder_mem <= reorder_mem_next;
         end if;        
      end if;
   end process;

OUTPUT_DECODE_IN: process (state_in, last_flag, ready_flag, in_enable, in_data, reorder_mem)
   begin
		next_mem_w <= '0';
		next_in_ready <= '0';
		reorder_mem_next <= reorder_mem; 
		
      case (state_in) is
			when st0_in =>
				next_in_ready <= '1';
         when st1_in =>
				next_in_ready <= '1';
				reorder_mem_next(23 downto 18) <= in_data(31 downto 26);
				reorder_mem_next(17 downto 12) <= in_data(23 downto 18);
				reorder_mem_next(11 downto  6) <= in_data(15 downto 10);
				reorder_mem_next( 5 downto  0) <= in_data( 7 downto  2);
         when st2_in =>
				if last_flag = '1' then
					next_in_ready <= not in_enable;
				else
					next_in_ready <= '1';
				end if;
				reorder_mem_next(47 downto 42) <= in_data(31 downto 26);
				reorder_mem_next(41 downto 36) <= in_data(23 downto 18);
				reorder_mem_next(35 downto 30) <= in_data(15 downto 10);
				reorder_mem_next(29 downto 24) <= in_data( 7 downto  2);
			when st3_in =>
				next_in_ready <= not in_enable;
				reorder_mem_next(71 downto 66) <= in_data(31 downto 26);
				reorder_mem_next(65 downto 60) <= in_data(23 downto 18);
				reorder_mem_next(59 downto 54) <= in_data(15 downto 10);
				reorder_mem_next(53 downto 48) <= in_data( 7 downto  2);
         when st4_in =>
				next_in_ready <= ready_flag;
				next_mem_w <= ready_flag;
         when st5_in =>
				next_in_ready <= ready_flag;
				next_mem_w <= ready_flag;				
			when others =>
      end case;  
   end process;

NEXT_STATE_DECODE_IN: process (state_in, last_flag, ready_flag, in_enable)
   begin
      next_state_in <= state_in;
      case (state_in) is
			when st0_in =>
				if ready_flag ='1' then
					next_state_in  <= st1_in;
				end if;
         when st1_in =>
				if in_enable='1' then 
					next_state_in <= st2_in;
				end if;
         when st2_in =>
				if in_enable = '1' then
					if last_flag = '1' then
						next_state_in <= st5_in;
					else
						next_state_in <= st3_in;
					end if;
				end if;
			when st3_in =>
				if in_enable= '1' then
					next_state_in <= st4_in;
				end if;
         when st4_in =>	
				if ready_flag = '1' then
					next_state_in <= st1_in;
				end if;
         when st5_in =>
				if ready_flag = '1' then
					next_state_in <= st1_in;
				end if;
         when others =>
      end case;
	end process;

------------------------------------------------------------------------------------
------------------------------------------------------------------------------------
	
SYNC_PROC_CO: process (clk)
   begin
      if (clk'event and clk = '1') then
         if (reset = '1') then
            state_co <= st0_co;
				in_x <= (others => '0');
				in_y <= (others => '0');
         else
            state_co <= next_state_co;
				in_x <= next_in_x;
				in_y <= next_in_y;
         end if;        
      end if;
   end process;
	
OUTPUT_DECODE_CO: process (state_co, in_x, out_x, in_y, out_y, mem_w)
   begin
		next_in_x  <= in_x;		
		next_in_y  <= in_y;		
		ready_flag <= '0';
		overload <= '0';		
		
		last_flag <= '0';
		
		case (state_co) is
			when st0_co =>
				if mem_w = '1' then
					next_in_x <= in_x + 1;					
				end if;
				ready_flag <= '1';
         when st1_co =>
				last_flag <= '1';
				ready_flag <= '1';
         when st2_co =>
				if in_y + 1 = out_y then
					overload <= '1';
				end if;
			when st3_co =>
				next_in_y <= in_y + 1;
				next_in_x <= (others => '0');
         when others =>
      end case;
		
   end process;
 
NEXT_STATE_DECODE_CO: process (state_co, in_x, mem_w, in_y, out_y)
   begin
      next_state_co <= state_co; 
      case (state_co) is
			when st0_co =>
				if in_x = 61 and mem_w = '1' then
					next_state_co <= st1_co;
				end if;
         when st1_co =>
				if mem_w = '1' then
					next_state_co <= st2_co;
				end if;
         when st2_co => 
				if in_y + 1 = out_y then
					-- overload
				else
					next_state_co <= st3_co;
				end if;
			when st3_co =>            
				next_state_co <= st0_co;
         when others =>
      end case;      
   end process;
	
------------------------------------------------------------------------------------
------------------------------------------------------------------------------------

SYNC_PROC_OUT: process (clk)
   begin
      if (clk'event and clk = '1') then
         if (reset = '1') then
            state_out <= st0_out; 
				out_ready <= '0';
				out_x<= (others => '0');
				out_y<= (others => '0');
         else
            state_out <= next_state_out;
				out_ready <= next_out_ready;
				out_x <= next_out_x;
				out_y <= next_out_y;
         end if;        
      end if;
   end process;
	
OUTPUT_DECODE_OUT: process (state_out, out_x, out_y, in_x, in_y, out_next)
   begin
		next_out_ready <= '0';
		next_out_x <= out_x;
		next_out_y <= out_y;
		UNDERLOAD <= '0';
		
		case (state_out) is
			when st0_out =>
			when st1_out =>
					next_out_ready <= '1';
					if out_next = '1' then
						next_out_x <= out_x + 1;
					end if;
         when st2_out => 
					if  out_x + 1 < in_x then
						next_out_ready <= '1';
						if out_next = '1' then
							next_out_x <= out_x + 1;
							if out_x + 2 = in_x then
								next_out_ready <= '0';
							end if;
						end if;
					end if;
			when st3_out =>
				if in_y = out_y then
					UNDERLOAD <= '1';
					if out_x = 53 then
						next_out_ready <= '1';
						if out_next = '1' then
							next_out_x <= out_x + 1;			-- wtf
						end if;
					end if;
				else
					next_out_y <= out_y + 1;
					next_out_x <= (others => '0');
				end if;
			when st4_out =>
				next_out_y <= out_y + 1;
				next_out_x <= (others => '0');
         when others =>
      end case;   
		
   end process;
 
NEXT_STATE_DECODE_OUT: process (state_out, in_y, out_y, in_x, out_x)
   begin
      next_state_out <= state_out; 
      case (state_out) is
			when st0_out =>
				if in_y = out_y then
					next_state_out <= st2_out;
				else
					next_state_out <= st1_out;
				end if;				
         when st1_out =>
				if out_x = 54 then
					next_state_out <= st4_out;
				end if;
         when st2_out => 
				if out_x = 53 then
					next_state_out <= st3_out;
				end if;				
				if in_y /= out_y then
					next_state_out <= st1_out;
				end if;
			when st3_out =>            
				if in_y = out_y then
					-- UNDERLOAD
				else
					next_state_out <= st0_out;
				end if;
			when st4_out =>
				next_state_out <= st0_out;
         when others =>
      end case;      
   end process;

----------------------------------------------------------------------------
----------------------------------------------------------------------------

mem_addr_w <=  in_y &  in_x;
mem_addr_r <= out_y & out_x;

mem: for N in 0 to 1 generate	
	bram: RAMB16BWER
		generic map (
			DATA_WIDTH_A => 36, 
			DATA_WIDTH_B => 36, 
			WRITE_MODE_A => "NO_CHANGE",
			WRITE_MODE_B => "NO_CHANGE",
			SIM_COLLISION_CHECK => "WARNING_ONLY",
			SIM_DEVICE => "SPARTAN6"
		)
		port map (	
		DIPA (3 downto 0) => reorder_mem(35 + N*36 downto 32 + N*36),
		DIA (31 downto 0) => reorder_mem(31 + N*36 downto  0 + N*36),		
		
		
		ADDRA(13 downto 5) => mem_addr_w,
		ADDRA( 4 downto 0) =>(others => '0'),
		WEA(0)	=> mem_w,
		WEA(1)	=> mem_w,
		WEA(2)	=> mem_w,
		WEA(3)	=> mem_w,
		ENA	=> '1',
		RSTA	=> '0',
		REGCEA=> '0',
		CLKA	=> clk, 
		
		DIB   => (others => '0'),
		DIPB  => (others => '0'),
		ADDRB(13 downto 5) => mem_addr_r,
		ADDRB( 4 downto 0) =>(others => '0'),
		WEB	=> (others => '0'),
		ENB	=> '1',
		RSTB	=> '0',
		REGCEB=> '0',
		CLKB	=> clk,
		
		DOA  => open,
		DOPA => open,
		
		DOPB (3 downto 0) => out_data(35 + N*36 downto 32 + N*36),
		DOB (31 downto 0) => out_data(31 + N*36 downto  0 + N*36)
		
	);
end generate;	
	
end Behavioral;

