----------------------------------------------------------------------------------
-- Detektor objektu
-- Copyright (C) 2012 Brno University of Technology,
--                        Faculty of Information Technology
-- Author(s): Petr Musil <xmusilpetr AT fit.vutbr.cz>
--  
-- LICENSE TERMS
--  
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 	1. Redistributions of source code must retain the above copyright
-- 	notice, this list of conditions and the following disclaimer.
-- 	2. Redistributions in binary form must reproduce the above copyright
-- 	notice, this list of conditions and the following disclaimer in
-- 	the documentation and/or other materials provided with the
-- 	distribution.
-- 	3. All advertising materials mentioning features or use of this software
-- 	or firmware must display the following acknowledgement:
--   
-- 	This product includes software developed by the University of
-- 	Technology, Faculty of Information Technology, Brno and its
-- 	contributors.
--  
-- 	4. Neither the name of the Company nor the names of its contributors
-- 	may be used to endorse or promote products derived from this
-- 	software without specific prior written permission.
--  
-- This software or firmware is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
-- 
--      $Id$
----------------------------------------------------------------------------------


-- without scale

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.std_logic_unsigned.all;
use work.configure.all;		-- my data types
use work.data_types.all;		-- my data types
use IEEE.NUMERIC_STD.ALL;
library UNISIM;
use UNISIM.VComponents.all;


entity stripe_memory is
 port(
		CLK	: in std_logic;
		SCALE_RESET	: in std_logic;
		OP		: in std_logic_vector(1 downto 0);
		
		------------ first port
		ADDR_A_X		: in std_logic_vector(9 downto 0);
		ADDR_A_Y		: in std_logic_vector(9 downto 0);
		ADDR_A_S		: in std_logic_vector(3 downto 0);	
		DATA_A		: out array_8x6x6b;
		
		------------ second port
		ADDR_B_X		: in std_logic_vector(9 downto 0);
		ADDR_B_Y		: in std_logic_vector(9 downto 0);
		ADDR_B_S		: in std_logic_vector(3 downto 0);
		DATA_B		: out array_8x6x6b;
		
		SCALE_READY	: out std_logic;
		
		W_DATA: in std_logic_vector(71 downto 0);	-- plnění
		W_E	: in std_logic;
		W_R	: out std_logic	
 );
end stripe_memory;

architecture Behavioral of stripe_memory is

component adress_engine is
	port(
		CLK	: in std_logic;
		
		ADDR_X	: in std_logic_vector(9 downto 0);
		ADDR_Y	: in std_logic_vector(9 downto 0);
		SCALE		: in std_logic_vector(3 downto 0);
		
		MEM_ADDR_A	: out array_12_9b;
		MEM_ADDR_B	: out array_12_9b;
		
		H_SHUFFLE	: out std_logic; 
		V_SHUFFLE	: out std_logic_vector(3 downto 0);
		C_SHUFFLE	: out std_logic_vector(2 downto 0);
		
		WRITE_SHUFFLE : out std_logic_vector(11 downto 0)		
	);
end component;
	
component mux_engine is
	port(
		CLK		: in std_logic;
		DATA_A	: in array_12x36b;
		DATA_B	: in array_12x36b;
		H_SHUFFLE	: in std_logic;
		V_SHUFFLE	: in std_logic_vector(3 downto 0);
		C_SHUFFLE	: in std_logic_vector(2 downto 0);
		
		DATA_OUT	: out array_8x6x6b
	);
end component;

component scaler4_5 is
    Port ( 
		CLK 				: in  STD_LOGIC;
		RESET 			: in STD_LOGIC;
      DATA_IN_A		: in array_8x6x6b;
		DATA_IN_VALID	: in std_logic;
		
		DATA_OUT			: out array_4x36b;
		DATA_OUT_READY : out std_logic
	);
end component;
	
signal MEM_ADDR_A_PA : array_12_9b;
signal MEM_ADDR_B_PA : array_12_9b;
signal H_SHUFFLE_PA, NEXT_H_SHUFFLE_PA : std_logic;
signal V_SHUFFLE_PA, NEXT_V_SHUFFLE_PA : std_logic_vector(3 downto 0);
signal C_SHUFFLE_PA, NEXT_C_SHUFFLE_PA : std_logic_vector(2 downto 0);
signal WRITE_SHUFFLE_PA : std_logic_vector(11 downto 0);

signal MEM_ADDR_A_PB : array_12_9b;
signal MEM_ADDR_B_PB : array_12_9b;
signal H_SHUFFLE_PB, NEXT_H_SHUFFLE_PB : std_logic;
signal V_SHUFFLE_PB, NEXT_V_SHUFFLE_PB : std_logic_vector(3 downto 0);
signal C_SHUFFLE_PB, NEXT_C_SHUFFLE_PB : std_logic_vector(2 downto 0);

type array_4x72b is array (3 downto 0) of std_logic_vector(71 downto 0);
signal write_ena_A	: std_logic_vector(11 downto 0);
signal write_ena_B	: std_logic_vector(11 downto 0);
signal next_write_ena_A	: std_logic_vector(11 downto 0);
signal next_write_ena_B	: std_logic_vector(11 downto 0);
signal next_w_data	: array_4x72b;
signal next_s_data	: array_4x36b;
signal cur_w_data		: std_logic_vector(71 downto 0);
signal next_op			: std_logic_vector( 1 downto 0);
signal next_WR			: std_logic;

signal DATA_A_PA		: array_12x36b;
signal DATA_B_PA		: array_12x36b;
signal DATA_A_PB		: array_12x36b;
signal DATA_B_PB		: array_12x36b;

signal data_a_out		: array_8x6x6b;
signal data_a_valid	: std_logic;
signal scale_valid ,scale_valid_next ,scale_valid_next_next	: std_logic;
signal scale_out		: array_4x36b;
signal scale_rdy	: std_logic;
signal scale_ready_12	: std_logic_vector(11 downto 0);

begin

SYNC_PROC: process (CLK)
   begin
      if (CLK'event and CLK = '1') then
			cur_w_data <= W_DATA;
			next_s_data <= scale_out;
			next_op <= OP;
			NEXT_H_SHUFFLE_PA <= H_SHUFFLE_PA;
			NEXT_H_SHUFFLE_PB <= H_SHUFFLE_PB;
			NEXT_V_SHUFFLE_PA <= V_SHUFFLE_PA;
			NEXT_V_SHUFFLE_PB <= V_SHUFFLE_PB;
			NEXT_C_SHUFFLE_PA <= C_SHUFFLE_PA;
			NEXT_C_SHUFFLE_PB <= C_SHUFFLE_PB;
			
			scale_valid_next <= scale_valid_next_next;
			scale_valid <= scale_valid_next;
		end if;
end process;

op_proc: process(op, next_op, WRITE_SHUFFLE_PA, W_DATA, scale_out, scale_ready_12,next_s_data, scale_rdy)
	begin
		W_R <='0';
		for i in 0 to 3 loop
			next_w_data(i) <= (others => '0');
		end loop;
		
		SCALE_READY <= scale_rdy;
		write_ena_A  <= (others => '0');
		write_ena_B  <= (others => '0');
		scale_valid_next_next <='0';
		
		case (next_op) is
			when "00"   =>
			when "01"  	=>
				scale_valid_next_next <='1';
			when "10" 	=>	
				if H_SHUFFLE_PA = '0' then
					write_ena_A <= scale_ready_12 and (WRITE_SHUFFLE_PA or WRITE_SHUFFLE_PA(10 downto 0)&'0' or WRITE_SHUFFLE_PA(9 downto 0)&"00" or WRITE_SHUFFLE_PA(8 downto 0)&"000");				
					for i in 0 to 3 loop
						next_w_data(i)(35 downto 0) <= next_s_data(i);
					end loop;
				else
					write_ena_B <= scale_ready_12 and (WRITE_SHUFFLE_PA or WRITE_SHUFFLE_PA(10 downto 0)&'0' or WRITE_SHUFFLE_PA(9 downto 0)&"00" or WRITE_SHUFFLE_PA(8 downto 0)&"000");					
					for i in 0 to 3 loop
						next_w_data(i)(71 downto 36) <= next_s_data(i);
					end loop;
				end if;				
			when "11"	=>
				W_R <='1';
				write_ena_A <= WRITE_SHUFFLE_PA;
				write_ena_B <= WRITE_SHUFFLE_PA;	
				for i in 0 to 3 loop
					next_w_data(i) <= cur_w_data;
				end loop;
			when others	=>
		end case;	
	end process;


map_addr_A: adress_engine port map(
	CLK	=> CLK,
	ADDR_X => ADDR_A_X,
	ADDR_Y => ADDR_A_Y,
	SCALE  => ADDR_A_S,
	MEM_ADDR_A => MEM_ADDR_A_PA,
	MEM_ADDR_B => MEM_ADDR_B_PA,
	H_SHUFFLE  => H_SHUFFLE_PA,
	V_SHUFFLE  => V_SHUFFLE_PA,
	C_SHUFFLE  => C_SHUFFLE_PA,
	WRITE_SHUFFLE => WRITE_SHUFFLE_PA
);

map_addr_B: adress_engine port map(
	CLK	=> CLK,
	ADDR_X => ADDR_B_X,
	ADDR_Y => ADDR_B_Y,
	SCALE  => ADDR_B_S,
	MEM_ADDR_A => MEM_ADDR_A_PB,
	MEM_ADDR_B => MEM_ADDR_B_PB,
	H_SHUFFLE  => H_SHUFFLE_PB,
	V_SHUFFLE  => V_SHUFFLE_PB,
	C_SHUFFLE  => C_SHUFFLE_PB,
	WRITE_SHUFFLE => open
);

map_mux_A: mux_engine port map(
		CLK		=> CLK,
		DATA_A	=> DATA_A_PA,
		DATA_B	=> DATA_B_PA,
		H_SHUFFLE=> NEXT_H_SHUFFLE_PA,
		V_SHUFFLE=> NEXT_V_SHUFFLE_PA,
		C_SHUFFLE=> NEXT_C_SHUFFLE_PA,	
		DATA_OUT	=> data_a_out
);

DATA_A <= data_a_out;

map_mux_B: mux_engine port map(
		CLK		=> CLK,
		DATA_A	=> DATA_A_PB,
		DATA_B	=> DATA_B_PB,
		H_SHUFFLE=> NEXT_H_SHUFFLE_PB,
		V_SHUFFLE=> NEXT_V_SHUFFLE_PB,
		C_SHUFFLE=> NEXT_C_SHUFFLE_PB,	
		DATA_OUT	=> DATA_B
);

mem_0: for Y in 0 to 11 generate
		bram_0: RAMB16BWER
			generic map (
				DATA_WIDTH_A => 36, 
				DATA_WIDTH_B => 36, 
				WRITE_MODE_A => "WRITE_FIRST",
				WRITE_MODE_B => "WRITE_FIRST",
				SIM_DEVICE => "SPARTAN6"
			)
			
			port map (
			DOA  => DATA_A_PA(Y)(31 downto 0),       -- Output port-A data			
			DOPA => DATA_A_PA(Y)(35 downto 32),
			
			DOB  => DATA_A_PB(Y)(31 downto 0),		
			DOPB => DATA_A_PB(Y)(35 downto 32),
			
			ADDRA(13 downto 5) => MEM_ADDR_A_PA(Y),
			ADDRA(4 downto 0)  => "00000",   -- Input port-A address
			ADDRB(13 downto 5) => MEM_ADDR_A_PB(Y),
			ADDRB(4 downto 0)  =>"00000",   -- Input port-B address
			CLKA => CLK,     -- Input port-A clock
			CLKB => CLK,     -- Input port-B clock
			
			DIA => next_w_data(Y mod 4)(31 downto 0),       -- Input port-A data
			DIPA=> next_w_data(Y mod 4)(35 downto 32),
			DIB => (others => '0'),       -- Input port-B data
			DIPB=> (others => '0'),
			ENA => '1',       -- Input port-A enable
			ENB => '1',       -- Input port-B enable
			REGCEA => '0', -- Input port-A output register enable
			REGCEB => '0', -- Input port-B output register enable
			RSTA => '0',     -- Input port-A reset
			RSTB => '0',     -- Input port-B reset
			WEA(0) => write_ena_A(Y),       -- Input port-A write enable
			WEA(1) => write_ena_A(Y),
			WEA(2) => write_ena_A(Y),
			WEA(3) => write_ena_A(Y),
			WEB => (others => '0')        -- Input port-B write enable
			);
end generate;

mem_1: for Y in 0 to 11 generate
		bram_0: RAMB16BWER
			generic map (
				DATA_WIDTH_A => 36, 
				DATA_WIDTH_B => 36, 
				WRITE_MODE_A => "WRITE_FIRST",
				WRITE_MODE_B => "WRITE_FIRST",
				SIM_DEVICE => "SPARTAN6"
			)
			
			port map (
			DOA  => DATA_B_PA(Y)(31 downto 0),       -- Output port-A data			
			DOPA => DATA_B_PA(Y)(35 downto 32),
			
			DOB  => DATA_B_PB(Y)(31 downto 0),		
			DOPB => DATA_B_PB(Y)(35 downto 32),
			
			ADDRA(13 downto 5) => MEM_ADDR_B_PA(Y),
			ADDRA(4 downto 0)  => "00000",   -- Input port-A address
			ADDRB(13 downto 5) => MEM_ADDR_B_PB(Y),
			ADDRB(4 downto 0)  =>"00000",   -- Input port-B address
			CLKA => CLK,     -- Input port-A clock
			CLKB => CLK,     -- Input port-B clock
			
			DIA => next_w_data(Y mod 4)(67 downto 36),       -- Input port-A data
			DIPA=> next_w_data(Y mod 4)(71 downto 68),
			DIB => (others => '0'),       -- Input port-B data
			DIPB=> (others => '0'),
			ENA => '1',       -- Input port-A enable
			ENB => '1',       -- Input port-B enable
			REGCEA => '0', -- Input port-A output register enable
			REGCEB => '0', -- Input port-B output register enable
			RSTA => '0',     -- Input port-A reset
			RSTB => '0',     -- Input port-B reset
			WEA(0) => write_ena_B(Y),       -- Input port-A write enable
			WEA(1) => write_ena_B(Y),
			WEA(2) => write_ena_B(Y),
			WEA(3) => write_ena_B(Y),
			WEB => (others => '0')        -- Input port-B write enable
			);
end generate;

scale: scaler4_5 port map( 
		CLK 				=> CLK, 
		RESET 			=> SCALE_RESET,
      DATA_IN_A		=> data_a_out,
		DATA_IN_VALID	=> scale_valid,		
		DATA_OUT			=> scale_out,
		DATA_OUT_READY => scale_rdy
	);

scale_ready_12<=scale_rdy&scale_rdy&scale_rdy&scale_rdy&scale_rdy&scale_rdy&scale_rdy&scale_rdy&scale_rdy&scale_rdy&scale_rdy&scale_rdy;

end Behavioral;

