----------------------------------------------------------------------------------
-- Detektor objektu
-- Copyright (C) 2012 Brno University of Technology,
--                        Faculty of Information Technology
-- Author(s): Petr Musil <xmusilpetr AT fit.vutbr.cz>
--  
-- LICENSE TERMS
--  
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 	1. Redistributions of source code must retain the above copyright
-- 	notice, this list of conditions and the following disclaimer.
-- 	2. Redistributions in binary form must reproduce the above copyright
-- 	notice, this list of conditions and the following disclaimer in
-- 	the documentation and/or other materials provided with the
-- 	distribution.
-- 	3. All advertising materials mentioning features or use of this software
-- 	or firmware must display the following acknowledgement:
--   
-- 	This product includes software developed by the University of
-- 	Technology, Faculty of Information Technology, Brno and its
-- 	contributors.
--  
-- 	4. Neither the name of the Company nor the names of its contributors
-- 	may be used to endorse or promote products derived from this
-- 	software without specific prior written permission.
--  
-- This software or firmware is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
-- 
--      $Id$
----------------------------------------------------------------------------------


library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use work.configure.all;		-- my data types
use work.data_types.all;		-- my data types
use ieee.std_logic_signed.all;


entity waldboost_detector is
			Port(	clk 					: in std_logic;
					reset 				: in std_logic;
					filter_enable		: in std_logic;
					led_overload		: out std_logic;
					led_underload		: out std_logic;
					bussy					: out std_logic;
					reg_out				: out std_logic_vector(31 downto 0);
								
					--pametovy vstup
					rd_data				: in std_logic_vector(31 downto 0);
					rd_data_valid		: in std_logic;
					rd_enable			: out std_logic;
					rd_addr				: out std_logic_vector(31 downto 0);
					rd_len				: out std_logic_vector(31 downto 0);
				
					--pametovy vystup
					wr_data				: out std_logic_vector(31 downto 0);
					wr_enable			: out std_logic;
					wr_busy				: out std_logic;
					wr_ready				: in std_logic;
					wr_addr			   : out std_logic_vector(31 downto 0)
					);
end waldboost_detector;

architecture Behavioral of waldboost_detector is

component instruct_memory is
	port(
		CLK		: in std_logic;
		ADDR_A	: in std_logic_vector(9 downto 0);
		ADDR_B	: in std_logic_vector(9 downto 0);
		
		DATA_OUT_A	: out std_logic_vector(35 downto 0);
		DATA_OUT_B	: out std_logic_vector(35 downto 0);
		
		W_DATA	: in std_logic_vector(35 downto 0);	-- plnění
		W_ADDR	: in std_logic_vector(9 downto 0);
		W_ENA		: in std_logic
	);
end component;

component convert12x8to12x6 is
	port(
		clk 			: in std_logic;
		reset			: in std_logic;
		
		-- vstup
		in_data		: in std_logic_vector(31 downto 0);
		in_enable	: in std_logic;
		
		-- vystup
		out_data		: out std_logic_vector(71 downto 0);
		out_valid	: out std_logic
		
	);
end component;

component stripe_memory is
 port(
		CLK	: in std_logic;
		SCALE_RESET	: in std_logic;
		OP		: in std_logic_vector(1 downto 0);
		
		------------ first port
		ADDR_A_X		: in std_logic_vector(9 downto 0);
		ADDR_A_Y		: in std_logic_vector(9 downto 0);
		ADDR_A_S		: in std_logic_vector(3 downto 0);	
		DATA_A		: out array_8x6x6b;
		
		------------ second port
		ADDR_B_X		: in std_logic_vector(9 downto 0);
		ADDR_B_Y		: in std_logic_vector(9 downto 0);
		ADDR_B_S		: in std_logic_vector(3 downto 0);
		DATA_B		: out array_8x6x6b;		
		
		-- plnění
		W_DATA: in std_logic_vector(71 downto 0);	
		W_E	: in std_logic;
		W_R	: out std_logic	
 );
end component;

component dsp is
	port(
		CLK				: in std_logic;
		OPTMODE			: in std_logic_vector(1 downto 0);				
		DATA_IN			: in array_6x6x6b;				
		DATA_OUT			: out dsp_out				
	);
end component;

component extractor is
	port(
		CLK 		: in std_logic;
		OPTMODE	: in feature_mode;
		
		TABLE_A	: in std_logic_vector(9 downto 0);	--použitá tabulka
		LBPMASK_A	: in std_logic_vector(2 downto 0);	--maska pro LBP příznak, 8 možností
		RANKA_A		: in std_logic_vector(3 downto 0);
		RANKB_A		: in std_logic_vector(3 downto 0);
		
		TABLE_B	: in std_logic_vector(9 downto 0);	--použitá tabulka
		LBPMASK_B	: in std_logic_vector(2 downto 0);	--maska pro LBP příznak, 8 možností
		RANKA_B		: in std_logic_vector(3 downto 0);
		RANKB_B		: in std_logic_vector(3 downto 0);
		
		DATA_IN_A	: in dsp_out;
		DATA_IN_B	: in dsp_out;
		
		DATA_OUT_A	: out std_logic_vector(8 downto 0);
		DATA_OUT_B	: out std_logic_vector(8 downto 0)		
	);
end component;

component treshold_memory18 is
	port(
		CLK	: in std_logic;
		ADDR_A	: in std_logic_vector( 9 downto 0 );
		ADDR_B	: in std_logic_vector( 9 downto 0 );
		
		DATA_OUT_A	: out std_logic_vector( 17 downto 0 );
		DATA_OUT_B	: out std_logic_vector( 17 downto 0 );
		
		W_ADDR	: in std_logic_vector( 9 downto 0) ;
		W_DATA	: in std_logic_vector( 17 downto 0 );
		W_ENA		: in std_logic
	);
end component;

component treshold_unit is
	port(
		CLK 		: in std_logic;
		RESET		: in std_logic;		
		INDEX		: in std_logic_vector(3 downto 0);
		TRESHOLD	: in std_logic_vector(17 downto 0);
		FEATURE	: in std_logic_vector(8 downto 0);		
		FIRST		: in std_logic;
		ENABLE	: in std_logic;		
		DETECT	: out std_logic;
		SUMA_OUT	: out std_logic_vector(17 downto 0)
		
	);
end component;

component detect_control is
	port(
		CLK		: in std_logic;
		RESET		: in std_logic;
		ENABLE	: in std_logic;
		BUFFER_READY: in std_logic;
		MEM_VALID 	: in std_logic;
		DETECT_A	: in std_logic;
		DETECT_B	: in std_logic;
		
		BUSSY		: out std_logic;
		INIT		: out std_logic;
		SCALE_RESET	: out std_logic;
		
		TABLE_A	: out std_logic_vector(9 downto 0);
		POS_X_A	: out std_logic_vector(9 downto 0);
		POS_Y_A	: out std_logic_vector(9 downto 0);
		POS_S_A	: out std_logic_vector(3 downto 0);
		INDEX_A	: out std_logic_vector(3 downto 0);
		ENA_A		: out std_logic;
		FIRST_A	: out std_logic;
		MODE_A	: out std_logic_vector(1 downto 0);
		
		TABLE_B	: out std_logic_vector(9 downto 0);
		POS_X_B	: out std_logic_vector(9 downto 0);
		POS_Y_B	: out std_logic_vector(9 downto 0);
		POS_S_B	: out std_logic_vector(3 downto 0);
		INDEX_B	: out std_logic_vector(3 downto 0);
		ENA_B		: out std_logic;
		FIRST_B	: out std_logic;
		
		POSITIVE_A	: out std_logic;
		POSITIVE_B	: out std_logic;		
		
		READ_ADDR	: out std_logic_vector(31 downto 0);
		READ_LEN		: out std_logic_vector(31 downto 0);
		READ_ENABLE	: out std_logic
	);
end component;

component output_buffer is
	port(
		CLK 	: in std_logic;
		RESET : in std_logic;
		OVERLOAD : out std_logic;
		
		----- input
		X_A	: in std_logic_vector(9 downto 0);
		Y_A	: in std_logic_vector(9 downto 0);
		S_A	: in std_logic_vector(3 downto 0);
		SUM_A	: in std_logic_vector(17 downto 0);
		ENA_A	: in std_logic;
		
		X_B	: in std_logic_vector(9 downto 0);
		Y_B	: in std_logic_vector(9 downto 0);
		S_B	: in std_logic_vector(3 downto 0);
		SUM_B	: in std_logic_vector(17 downto 0);
		ENA_B	: in std_logic;
		
		----- output
		OUT_DATA	: out std_logic_vector(31 downto 0);
		OUT_ENA	: out std_logic;
		OUT_BUSSY: out std_logic;
		OUT_READY: in std_logic;
		OUT_ADDR	: out std_logic_vector(31 downto 0);
		OUT_COUNT : out std_logic_vector(31 downto 0)		
		
		);
end component;

component chipskopeB
  PORT (
    CONTROL : INOUT STD_LOGIC_VECTOR(35 DOWNTO 0);
    CLK : IN STD_LOGIC;
    TRIG0 : IN STD_LOGIC_VECTOR(9 DOWNTO 0);
    TRIG1 : IN STD_LOGIC_VECTOR(9 DOWNTO 0);
    TRIG2 : IN STD_LOGIC_VECTOR(9 DOWNTO 0);
    TRIG3 : IN STD_LOGIC_VECTOR(35 DOWNTO 0);
    TRIG4 : IN STD_LOGIC_VECTOR(7 DOWNTO 0);
    TRIG5 : IN STD_LOGIC_VECTOR(0 TO 0);
    TRIG6 : IN STD_LOGIC_VECTOR(0 TO 0);
    TRIG7 : IN STD_LOGIC_VECTOR(31 DOWNTO 0);
    TRIG8 : IN STD_LOGIC_VECTOR(71 DOWNTO 0);
    TRIG9 : IN STD_LOGIC_VECTOR(31 DOWNTO 0);
    TRIG10 : IN STD_LOGIC_VECTOR(31 DOWNTO 0);
    TRIG11 : IN STD_LOGIC_VECTOR(0 TO 0);
    TRIG12 : IN STD_LOGIC_VECTOR(0 TO 0));

end component;

component chipsicon
  PORT (
    CONTROL0 : INOUT STD_LOGIC_VECTOR(35 DOWNTO 0));

end component;

signal control_wire  :  STD_LOGIC_VECTOR(35 DOWNTO 0);

signal image_reset 	: std_logic;
type array_9x36b is array (9 downto 1) of std_logic_vector(35 downto 0);
type array_9x10b is array (9 downto 0) of std_logic_vector( 9 downto 0);
type array_2x10b is array (1 downto 0) of std_logic_vector( 9 downto 0);
type array_9x4b  is array (9 downto 0) of std_logic_vector( 3 downto 0);
type array_2x2b  is array (1 downto 0) of std_logic_vector( 1 downto 0);
signal instrucs_A		: array_9x36b;
signal instrucs_B		: array_9x36b;
signal table_A			: array_9x10b;
signal table_B			: array_9x10b;
signal posX_A,posX_B	: array_9x10b;
signal posY_A,posY_B	: array_9x10b;
signal index_A			: array_9x4b;
signal index_B			: array_9x4b;
signal opmode_A		: array_2x2b;
signal posS_A,posS_B	: array_9x4b;
signal ena_detect_A	: std_logic_vector(8 downto 0);
signal ena_detect_B	: std_logic_vector(8 downto 0);
signal first_A			: std_logic_vector(8 downto 0);
signal first_B			: std_logic_vector(8 downto 0);

signal buffer_data	: std_logic_vector(71 downto 0);
signal buffer_ready	: std_logic;
signal buffer_next	: std_logic;
signal buffer_enable : std_logic;

signal image_block_A : array_8x6x6b;
signal image_block_B : array_8x6x6b;
signal dsp_data_A		: dsp_out;
signal dsp_data_B		: dsp_out;
signal feature_A		: std_logic_vector(8 downto 0);
signal feature_B		: std_logic_vector(8 downto 0);
signal treshold_A		: std_logic_vector(17 downto 0);
signal treshold_B		: std_logic_vector(17 downto 0);
signal detect_A		: std_logic;
signal detect_B		: std_logic;
signal detect_suma_A	: std_logic_vector(17 downto 0);
signal detect_suma_B	: std_logic_vector(17 downto 0);
signal positive_A		: std_logic;
signal positive_B		: std_logic;

signal posXs_A			: std_logic_vector(9 downto 0);
signal posYs_A			: std_logic_vector(9 downto 0);
signal posXs_B			: std_logic_vector(9 downto 0);
signal posYs_B			: std_logic_vector(9 downto 0);

signal image_buffer_ready	: std_logic;

signal debug			: std_logic_vector(7 downto 0);
signal addr_rd			: std_logic_vector(31 downto 0);
signal len_rd			: std_logic_vector(31 downto 0);
signal enable_rd		: std_logic;
signal scale_reset	: std_logic;

begin

SYNC_PROC_IN: process (clk, reset, table_A, table_B, instrucs_A, instrucs_B)
   begin
      if (clk'event and clk = '1') then
         if (reset = '1') then
         else			
				instrucs_A(2) <= instrucs_A(1);
				instrucs_A(3) <= instrucs_A(2);
				instrucs_A(4) <= instrucs_A(3);
				instrucs_A(5) <= instrucs_A(4);
				instrucs_A(6) <= instrucs_A(5);
				instrucs_A(7) <= instrucs_A(6);
				instrucs_A(8) <= instrucs_A(7);
				instrucs_A(9) <= instrucs_A(8);
				
				instrucs_B(2) <= instrucs_B(1);
				instrucs_B(3) <= instrucs_B(2);
				instrucs_B(4) <= instrucs_B(3);
				instrucs_B(5) <= instrucs_B(4);
				instrucs_B(6) <= instrucs_B(5);
				instrucs_B(7) <= instrucs_B(6);
				instrucs_B(8) <= instrucs_B(7);
				instrucs_B(9) <= instrucs_B(8);
				
				table_A(1) <= table_A(0);
				table_A(2) <= table_A(1);
				table_A(3) <= table_A(2);
				table_A(4) <= table_A(3);
				table_A(5) <= table_A(4);
				table_A(6) <= table_A(5);
				table_A(7) <= table_A(6);
				table_A(8) <= table_A(7);
				table_A(9) <= table_A(8);
				
				table_B(1) <= table_B(0);
				table_B(2) <= table_B(1);
				table_B(3) <= table_B(2);
				table_B(4) <= table_B(3);
				table_B(5) <= table_B(4);
				table_B(6) <= table_B(5);
				table_B(7) <= table_B(6);
				table_B(8) <= table_B(7);
				table_B(9) <= table_B(8);
				
				index_A(1) <= index_A(0);
				index_A(2) <= index_A(1);
				index_A(3) <= index_A(2);
				index_A(4) <= index_A(3);
				index_A(5) <= index_A(4);
				index_A(6) <= index_A(5);
				index_A(7) <= index_A(6);
				index_A(8) <= index_A(7);
				index_A(9) <= index_A(8);
				
				index_B(1) <= index_B(0);
				index_B(2) <= index_B(1);
				index_B(3) <= index_B(2);
				index_B(4) <= index_B(3);
				index_B(5) <= index_B(4);
				index_B(6) <= index_B(5);
				index_B(7) <= index_B(6);
				index_B(8) <= index_B(7);
				index_B(9) <= index_B(8);
				
				ena_detect_A(1) <= ena_detect_A(0);
				ena_detect_A(2) <= ena_detect_A(1);
				ena_detect_A(3) <= ena_detect_A(2);
				ena_detect_A(4) <= ena_detect_A(3);
				ena_detect_A(5) <= ena_detect_A(4);
				ena_detect_A(6) <= ena_detect_A(5);
				ena_detect_A(7) <= ena_detect_A(6);
				ena_detect_A(8) <= ena_detect_A(7);
				
				ena_detect_B(1) <= ena_detect_B(0);
				ena_detect_B(2) <= ena_detect_B(1);
				ena_detect_B(3) <= ena_detect_B(2);
				ena_detect_B(4) <= ena_detect_B(3);
				ena_detect_B(5) <= ena_detect_B(4);
				ena_detect_B(6) <= ena_detect_B(5);
				ena_detect_B(7) <= ena_detect_B(6);
				ena_detect_B(8) <= ena_detect_B(7);
				
				first_A(1) <= first_A(0);
				first_A(2) <= first_A(1);
				first_A(3) <= first_A(2);
				first_A(4) <= first_A(3);
				first_A(5) <= first_A(4);
				first_A(6) <= first_A(5);
				first_A(7) <= first_A(6);
				first_A(8) <= first_A(7);
			
				first_B(1) <= first_B(0);
				first_B(2) <= first_B(1);
				first_B(3) <= first_B(2);
				first_B(4) <= first_B(3);
				first_B(5) <= first_B(4);
				first_B(6) <= first_B(5);
				first_B(7) <= first_B(6);
				first_B(8) <= first_B(7);
				
				posX_A(1) <= posX_A(0);
				posX_A(2) <= posX_A(1);
				posX_A(3) <= posX_A(2);
				posX_A(4) <= posX_A(3);
				posX_A(5) <= posX_A(4);
				posX_A(6) <= posX_A(5);
				posX_A(7) <= posX_A(6);
				posX_A(8) <= posX_A(7);
				posX_A(9) <= posX_A(8);
				
				posY_A(1) <= posY_A(0);
				posY_A(2) <= posY_A(1);
				posY_A(3) <= posY_A(2);
				posY_A(4) <= posY_A(3);
				posY_A(5) <= posY_A(4);
				posY_A(6) <= posY_A(5);
				posY_A(7) <= posY_A(6);
				posY_A(8) <= posY_A(7);
				posY_A(9) <= posY_A(8);
				
				posX_B(1) <= posX_B(0);
				posX_B(2) <= posX_B(1);
				posX_B(3) <= posX_B(2);
				posX_B(4) <= posX_B(3);
				posX_B(5) <= posX_B(4);
				posX_B(6) <= posX_B(5);
				posX_B(7) <= posX_B(6);
				posX_B(8) <= posX_B(7);
				posX_B(9) <= posX_B(8);
				
				posY_B(1) <= posY_B(0);	
				posY_B(2) <= posY_B(1);
				posY_B(3) <= posY_B(2);
				posY_B(4) <= posY_B(3);
				posY_B(5) <= posY_B(4);
				posY_B(6) <= posY_B(5);
				posY_B(7) <= posY_B(6);
				posY_B(8) <= posY_B(7);
				posY_B(9) <= posY_B(8);
				
				posS_A(1) <= posS_A(0);
				posS_A(2) <= posS_A(1);
				posS_A(3) <= posS_A(2);
				posS_A(4) <= posS_A(3);
				posS_A(5) <= posS_A(4);
				posS_A(6) <= posS_A(5);
				posS_A(7) <= posS_A(6);
				posS_A(8) <= posS_A(7);
				posS_A(9) <= posS_A(8);
				
				posS_B(1) <= posS_B(0);
				posS_B(2) <= posS_B(1);
				posS_B(3) <= posS_B(2);
				posS_B(4) <= posS_B(3);
				posS_B(5) <= posS_B(4);
				posS_B(6) <= posS_B(5);
				posS_B(7) <= posS_B(6);
				posS_B(8) <= posS_B(7);
				posS_B(9) <= posS_B(8);
				
				opmode_A(1) <= opmode_A(0);			

         end if;        
      end if;
   end process;
 
posXs_A <= posX_A(1) + instrucs_A(1)(35 downto 31);
posYs_A <= posY_A(1) + instrucs_A(1)(30 downto 26);
posXs_B <= posX_B(1) + instrucs_B(1)(35 downto 31);
posYs_B <= posY_B(1) + instrucs_B(1)(30 downto 26);

instruct_map: instruct_memory port map(
	CLK		=> CLK,
	ADDR_A	=> table_A(0),
	ADDR_B	=> table_B(0),		
	DATA_OUT_A	=> instrucs_A(1),
	DATA_OUT_B	=> instrucs_B(1),	
	W_DATA	=> (others => '0'),	
	W_ADDR	=> (others => '0'),	
	W_ENA		=> '0'
	);

input_buffer: convert12x8to12x6 port map(
	clk   => clk,
	reset => image_reset,
	in_data	 => rd_data,
	in_enable => rd_data_valid,
	out_data  => buffer_data,
	out_valid => buffer_ready
);

image_memory: stripe_memory port map(
	CLK 	=> clk,
	SCALE_RESET => scale_reset,			
	op		=> opmode_A(1),
	ADDR_A_X	=>  posXs_A,
	ADDR_A_Y	=>  posYs_A,
	ADDR_A_S	=>	posS_A(1),
	DATA_A	=> image_block_A,
		
	ADDR_B_X	=>	 posXs_B,
	ADDR_B_Y	=>	 posYs_B,
	ADDR_B_S	=>	posS_B(1),
	DATA_B	=>	image_block_B,
		
	W_DATA => buffer_data,	
	W_E => buffer_enable,
	W_R => buffer_next
);

dsp_unit_A: dsp port map(
	CLK		=> clk,
	OPTMODE	=> instrucs_A(4)(25 downto 24),	
	DATA_IN(0)	=> image_block_A(0),
	DATA_IN(1)	=> image_block_A(1),
	DATA_IN(2)	=> image_block_A(2),
	DATA_IN(3)	=> image_block_A(3),
	DATA_IN(4)	=> image_block_A(4),
	DATA_IN(5)	=>	image_block_A(5),
	DATA_OUT	=> dsp_data_A	
);

dsp_unit_B: dsp port map(
	CLK		=> clk,
	OPTMODE	=>	instrucs_B(4)(25 downto 24),
	DATA_IN(0)	=> image_block_B(0),
	DATA_IN(1)	=> image_block_B(1),
	DATA_IN(2)	=> image_block_B(2),
	DATA_IN(3)	=> image_block_B(3),
	DATA_IN(4)	=> image_block_B(4),
	DATA_IN(5)	=>	image_block_B(5),
	DATA_OUT	=> dsp_data_B	
);

feature_unit : extractor port map(
	CLK 			=> CLK,
	OPTMODE		=> F_LRD,					-- todo
	TABLE_A		=> table_A(5),
	LBPMASK_A	=> (others => '0'),		-- todo
	RANKA_A		=> instrucs_A(5)(23 downto 20),
	RANKB_A		=> instrucs_A(5)(19 downto 16),
	TABLE_B		=> table_B(5),
	LBPMASK_B	=> (others => '0'),		-- todo
	RANKA_B		=> instrucs_B(5)(23 downto 20),
	RANKB_B		=> instrucs_B(5)(19 downto 16),	
	DATA_IN_A	=> dsp_data_A,		
	DATA_OUT_A	=> feature_A,
	DATA_IN_B	=> dsp_data_B,		
	DATA_OUT_B	=> feature_B			
);

treshold_mem_map: treshold_memory18 port map(
		CLK		=> CLK,
		ADDR_A	=> table_A(7),
		ADDR_B	=> table_B(7),		
		DATA_OUT_A	=> treshold_A,
		DATA_OUT_B	=> treshold_B,		
		W_ADDR	=> (others => '0'),		
		W_DATA	=> (others => '0'),		
		W_ENA		=> '0'						
	);

treshold_A_map: treshold_unit port map(
	CLK 		=> CLK,
	RESET		=> image_reset,		
	INDEX		=> index_A(8),
	TRESHOLD	=> treshold_A,		
	FEATURE	=> feature_A,		
	FIRST		=> first_A(8),	
	ENABLE	=> ena_detect_A(8),		
	DETECT	=> detect_A,
	SUMA_OUT => detect_suma_A
	);
	
treshold_B_map: treshold_unit port map(
	CLK 		=> CLK,
	RESET		=> image_reset,		
	INDEX		=> index_B(8),
	TRESHOLD	=> treshold_B,		
	FEATURE	=> feature_B,		
	FIRST		=> first_B(8),
	ENABLE	=> ena_detect_B(8),
	DETECT	=> detect_B,
	SUMA_OUT => detect_suma_B
	);

control: detect_control port map(
		CLK		=> CLK,
		RESET		=> reset,
		ENABLE	=> filter_enable,
		BUFFER_READY => buffer_ready,
		MEM_VALID => rd_data_valid,
		DETECT_A	=> detect_A,
		DETECT_B	=> detect_B,		
		BUSSY		=> bussy,
		INIT => image_reset,	
		SCALE_RESET	=> scale_reset,
		TABLE_A	=> table_A(0),
		POS_X_A	=> posX_A(0),
		POS_Y_A	=> posY_A(0),
		POS_S_A	=> posS_A(0),
		INDEX_A	=> index_A(0),
		ENA_A		=> ena_detect_A(0),
		FIRST_A	=> first_A(0),
		MODE_A	=> opmode_A(0),
		
		TABLE_B	=> table_B(0),	
		POS_X_B	=> posX_B(0),
		POS_Y_B	=> posY_B(0),
		POS_S_B	=> posS_B(0),
		INDEX_B	=> index_B(0),
		ENA_B		=> ena_detect_B(0),
		FIRST_B	=> first_B(0),
		
		POSITIVE_A	=> positive_A,
		POSITIVE_B	=> positive_B,		
		
		READ_ADDR	=> addr_rd,
		READ_LEN		=> len_rd,
		READ_ENABLE	=> enable_rd
		
	);
	
rd_len <= len_rd;
rd_addr <= addr_rd;
rd_enable <= enable_rd;
	
output: output_buffer 	port map (
		CLK 	=> CLK,
		RESET => image_reset,
		OVERLOAD => open,
		
		----- input
		X_A	=> posX_A(9),
		Y_A	=> posY_A(9),
		S_A	=> posS_A(9),
		SUM_A	=> detect_suma_A,
		ENA_A	=> positive_A,
		
		X_B	=> posX_B(9),
		Y_B	=> posY_B(9),
		S_B	=> posS_B(9),
		SUM_B	=> detect_suma_B,
		ENA_B	=> positive_B,
		
		OUT_DATA	=> wr_data,
		OUT_ENA	=> wr_enable,
		OUT_BUSSY => wr_busy,
		OUT_READY => wr_ready,
		OUT_ADDR	 => wr_addr,
		OUT_COUNT => reg_out
		);
		

end Behavioral;
