//==============================================================================
/*! \file
 * OpenMesh Toolkit for mesh analysis    \n
 * Copyright (c) 2010 by Rostislav Hulik     \n
 *
 * Author:  Rostislav Hulik, rosta.hulik@gmail.com  \n
 * Date:    2010/3/2                          \n
 *
 * This file is part of software developed for support of Rostislav Hulik's dissertation thesis at dcgm-robotics@FIT group.
 *
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this file.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Description:
 * - Class representing a triangle in ray casting R-tree structure
 */

#ifndef OM_BSP_TREE_TRIANGLE_H
#define OM_BSP_TREE_TRIANGLE_H

#include <OMToolkit\OMTypes.h>
#include <OMToolkit\OMTriBSPTreeRay.h>

namespace OMToolkit
{
	class TriCenter
	{
		public:

			/**
			 * Mesh Type
			 */
			typedef OMToolkit::Types::ModuleMeshd MeshT;
			
			/**
			 * Point Type
			 */
			typedef MeshT::Point PointT;

			/**
			 * Face handle Type
			 */
			typedef MeshT::FaceHandle FaceHT;

			/**
			 * Scalar type
			 */
			typedef MeshT::Scalar ScalarT;

			/**
			 * Void constructor
			 */
			TriCenter();

			/**
			 * Constructor - initializes a structure
			 * @param center - center of triangle
			 * @param face - Face handle in OM structure
			 * @param aabbLow - Lower values of triangle's AABB
			 * @param aabbHigh - Higher values of triangle's AABB
			 * @param mesh - pointer to mesh
			 */
			TriCenter(PointT &center, FaceHT face, PointT &aabbLow, PointT &aabbHigh, Types::ModuleMeshd *mesh);

			/**
			 * Tests intersection between ray and face (triangle)
			 * @param origin Ray origin - intersection is computed also in inverse direction
			 * @param vector Ray direction
			 * @param face Face handle
			 * @param distance Computed parameter of intersection
			 * @return True if face is hit
			 */
			bool intersects(OMRay &ray, ScalarT &distance);

			/**
			 * Center of triangle
			 */
			PointT m_center;

			/**
			 * Face handle in OM structure
			 */
			FaceHT m_face;

			/**
			 * Triangle's AABB
			 */
			PointT m_aabb[2];

			/**
			 * Points of triangle
			 */
			PointT m_points[3];

			/**
			 * Edges of triangle
			 */
			MeshT::Normal m_edge1, m_edge2;
};
}

#endif