//==============================================================================
/*! \file
 * OpenMesh Toolkit for mesh analysis    \n
 * Copyright (c) 2010 by Rostislav Hulik     \n
 *
 * Author:  Rostislav Hulik, rosta.hulik@gmail.com  \n
 * Date:    2010/10/25                          \n
 *
 * This file is part of software developed for support of Rostislav Hulik's dissertation thesis at dcgm-robotics@FIT group.
 *
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this file.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Module description:
 * - Gets a mesh from input channel
 * - Proceedes with mesh decimation with metric given by input arguments
 * - Writes an output mesh into output channel
 */

#ifndef _OM_DECIMATE_MESH_H_
#define _OM_DECIMATE_MESH_H_

#include <MDSTk/Module/mdsModule.h>

/**
 * Module reads a mesh, decimates it and writes it into output mesh
 * Uses methods implemented in OpenMesh, possible collapse metrics are:
 *	- Maximum roudness of triangle
 *	- Quadric error
 * Possible to switch on vertex locking (independent sets) or checking normal deviation
 */
class OMDecimateMesh : public mds::mod::CModule
{
	public:
		/**
		 * Smart pointer type
		 */
		MDS_SHAREDPTR(OMDecimateMesh);

		/**
		 * Constructor
		 * @param sDescription Description of the module
		 */
		OMDecimateMesh(const std::string& sDescription);

		/**
		 * Virtual destructor
		 */
		virtual ~OMDecimateMesh();

	protected:
		/**
		 * Virtual method called on startup
		 * @returns True if startup was successful
		 */
		virtual bool startup();

		/**
		 * Virtual method called by the processing thread
		 * @returns True if main loop continues
		 */
		virtual bool main();

		/**
		 * Called on console shutdown
		 */
		virtual void shutdown();

		/**
		 * Called on writing a usage statement
		 * @param Stream Output stream
		 */
		virtual void writeExtendedUsage(std::ostream& Stream);

		/**
		 * Used collapse metrics
		 */
		std::string m_collapse_metrics;

		/**
		 * Minimum roudness (roudness collapse)
		 */
		double m_minimum_roundness;

		/**
		 * Maximum error (error quadric)
		 */
		double m_max_error;

		/**
		 * Maximum normal deviation (normal deviation)
		 */
		double m_max_norm_dev;

		/**
		 * Number of collapses
		 */
		int m_max_collapses;

		/**
		 * Number of vertices
		 */
		int m_max_vertices;

		/**
		 * Use collapsed vertex locking
		 */
		bool m_independent_sets;
};

/**
 * Smart pointer to console application
 */
typedef OMDecimateMesh::tSmartPtr    OMDecimateMeshPtr;

#endif // _OM_DECIMATE_MESH_H_