//==============================================================================
/*! \file
 * OpenMesh Toolkit for mesh analysis    \n
 * Copyright (c) 2010 by Rostislav Hulik     \n
 *
 * Author:  Rostislav Hulik, rosta.hulik@gmail.com  \n
 * Date:    2010/10/20                          \n
 *
 * This file is part of software developed for support of Rostislav Hulik's dissertation thesis at dcgm-robotics@FIT group.
 *
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this file.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Description:
 * - Class for descriptor computation in mesh data
 */
#ifndef _COMPUTE_DESCRIPTORS_H_
#define _COMPUTE_DESCRIPTORS_H_

#include <OMToolkit\OMTypes.h>

namespace OMToolkit
{
	/**
	 * Class for descriptor computation in mesh data
	 */
	class OMMatrixDescriptors
	{
		public:
			/**
			 * Mesh type definition
			 */
			typedef Types::ModuleMeshd MeshT;
			
			/**
			 * Scalar type definition
			 */
			typedef MeshT::Scalar ScalarT;

			/**
			 * Normal type definition
			 */
			typedef MeshT::Normal NormalT;

			/**
			 * Attribute scalar definition
			 */
			typedef MeshT::AttributeScalar AScalarT;

			/**
			 * Matrix type definition
			 */
			typedef Types::OMSerializableMatrix<AScalarT> MatrixT;

			/**
			 * Constructor initializes this class for ongoing computations
			 * @param mesh Input mesh
			 */
			OMMatrixDescriptors(MeshT *mesh);

			/**
			 * The function computes feature points using pre-computed curvature on the mesh
			 * @param matrixH The matrix handle
			 * @param curvatureMagH The curvature magnitude handle
			 * @param finalNum The final number of feature points in the mesh
			 */
			bool ComputePoints(OpenMesh::VPropHandleT<MatrixT> matrixH, OpenMesh::VPropHandleT<AScalarT> curvatureMagH, int finalNum);

			/**
			 * The function extracts the descriptors from the points marked as feature points
			 * Format of the output:
			 *	Vector of the vertices features
			 *	Each row contains ID of vertex + 128 dimensional feature vector
			 * @see ComputePoints
			 * @param matrixH The matrix handle
			 * @param curvatureMagH The curvature magnitude handle
			 * @param matrix_size The size of matrix used for descriptor computation (relative to the median of edge lengths)
			 * @param matrix_resolution The resolution of the matrix
			 * @param desc_diameter The descriptor diameter
			 */
			std::vector<std::vector<float>> ComputeDescriptors(OpenMesh::VPropHandleT<MatrixT> matrixH, OpenMesh::VPropHandleT<AScalarT> curvatureMagH, float matrix_size, int matrix_resolution, float desc_diameter);
		
		private:
			/**
			 * Local pointer to the mesh
			 */
			MeshT *m_mesh;
	};
}

#endif