//==============================================================================
/*! \file
 * OpenMesh Toolkit for mesh analysis    \n
 * Copyright (c) 2010 by Rostislav Hulik     \n
 *
 * Author:  Rostislav Hulik, rosta.hulik@gmail.com  \n
 * Date:    2010/10/20                          \n
 *
 * This file is part of software developed for support of Rostislav Hulik's dissertation thesis at dcgm-robotics@FIT group.
 *
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this file.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Description:
 * - Class for simulating Gauss function
 */

#ifndef _OM_GAUSS_FUNC_H_
#define _OM_GAUSS_FUNC_H_

#include <stdlib.h>

namespace OMToolkit
{
	/**
	 * Gauss function class
	 */
	template <typename Scalar>
	class OMGaussFunction
	{
		public:
			/**
			 * Constructor - precomputes helper variables
			 * @param mu - Shift
			 * @param sigma - Variance
			 */
			OMGaussFunction(Scalar mu, Scalar sigma)
			{
				m_mu = mu;
				m_front = 1/(sigma * sqrt(2 * M_PI));
				m_denomExp = 2 * sigma * sigma;
			}

			/**
			 * Function returns normalized gauss function value for x
			 * @param x Variable X
			 * @return Gauss function value in x
			 */
			Scalar getValNorm(Scalar x)
			{
				Scalar aux = (x - m_mu);
				aux *= aux;
				return m_front * exp( (-aux) / m_denomExp);
			}

			/**
			 * Function returns NON normalized gauss function value for x
			 * @param x Variable X
			 * @return Gauss function value in x
			 */
			Scalar getVal(Scalar x)
			{
				Scalar aux = (x - m_mu);
				aux *= aux;
				return exp( (-aux) / m_denomExp);
			}

		private:

			/**
			 * Saved shift
			 */
			Scalar m_mu;

			/*
			 * Helper variable in gauss function computation
			 */
			Scalar m_front;

			/*
			 * Helper variable in gauss function computation
			 */
			Scalar m_denomExp;
	}; // class
}

#endif