/**
 *  Author: Jan Brejcha <ibrejcha@fit.vutbr.cz>, <brejchaja@gmail.com>
 *  Copyright (C) 2014  Jan Brejcha
 *
 *  OPEN SOURCE LICENCE VUT V BRNĚ
 *  Verze 1.
 *  Copyright (c) 2010, Vysoké učení technické v Brně, Antonínská 548/1, PSČ 601 90
 *  -------------------------------------------------------------------------------
 */

#ifndef CONTOURLETT_EXTRACTOR
#define CONTOURLETT_EXTRACTOR

#ifdef __APPLE__
    #include <OpenGL/gl.h>
#else
    #define GL_GLEXT_PROTOTYPES
    #include <GL/gl.h>
    #include <GL/glext.h>
#endif

#include <vector>
#include <cstdio>
#include <iostream>
#include <cmath>
#include <memory>
#include <algorithm>
#include <stdexcept>
#include <iterator>

#include <QString>
#include <QImage>
#include <QDataStream>
#include <QFileInfo>

#include "Contourlett.h"
#include "GaussianSampler.h"
#include "Image.h"
#include "Convolution.h"

#define PANORAMA_HEIGHT 4516
#define PANORAMA_HEIGHT_2 2258
/**
 * Extracts horizon line countourletts into vectors
 */
class ContourlettExtractor {
public:

    /**
     * @brief create
     * Factory method which creates CE either from image or from binary
     * horizon file based on file extension.
     * @param imgPath
     * @return new instance of ContourlettExtractor
     */
    static ContourlettExtractor create(QString imgPath);

    /**
     * Initializer of ContourlettExtractor. 
     * @param imgPath   the source path to silhouettes 360 degrees panorama image
     * @param cwidth    contourlett width in degrees, default value is 10 deg.
     */
    ContourlettExtractor(QString imgPath);

    /**
     * @brief ContourlettExtractor
     * Copy constructor.
     * @param other
     */
    ContourlettExtractor(const ContourlettExtractor &other);

    /**
     * @brief ContourlettExtractor Initializes ContourlettExtractor directly
     * from an image.
     * @param img Image to extract the horizon line and the features from.
     */
    ContourlettExtractor(QImage img);

    /**
     * @brief ContourlettExtractor Initializes ContourlettExtractor from
     * extracted horizon line. This initializer can be used to initialize
     * the object when loading horizon line from file.
     * @param imageWidth Width of the original image.
     * @param imageHeight Height of the original image.
     * @param horizonLine HorizonLine values.
     */
    ContourlettExtractor(int imageWidth, int imageHeight,
                                               double *horizonLine);


    
    ~ContourlettExtractor();
    
    /**
     * Extracts the contourletts from the source image. Sliding window is used 
     * for contourletts extraction.
     * 
     * @param fov		Field of view of the image in degrees. If it is query image,
     * 					set as fov of the image, if 360 degree panorama,
     * 					set 360.
     * @param cWidth	contourlett width in degrees
     */
    void extract(float fov, float cWidth, std::vector< std::shared_ptr<LocalFeature> > &res);
    
    /**
	 * Get the vector of points from the horizon line
     * @param fov field-of-view in degrees
     */
    std::vector< std::pair<int, int> > getHorizonLinePoints(float fov);

    std::vector< std::pair<int, int> > getHorizonLinePanoramaPoints();

    std::vector< std::pair<int, int> > getSingleHorizonLinePanoramaPoints();

    /**
     * @brief loadFromBin Loads the horizon to be extracted from binary format
     * and creates new ContourlettExtractor with this horizon.
     * @param filename File path pointing to the binary horizon file created by
     * saveToBin().
     * @return ContourlettExtractor object with the loaded horizon line from
     * binary file.
     */
    static ContourlettExtractor loadFromBin(QString filename);

    /**
     * @brief saveToBin Saves the horizon line into binary file format.
     * Can be loaded later by loadFromBin() to extract features from the
     * horizon.
     * @param filename File path to save the binary horizon line to.
     */
    void saveToBin(QString filename);

    /**
     * @brief width
     * @return Returns the width of original image.
     */
    int width();

    /**
     * @brief heigh
     * @return Returns the height of original image.
     */
    int height();

    /**
     * Convolves the horizon line with the gaussian to smoothen the signal
     * to avoid aliasing.
     */
    void smoothenHorizonLine();

private:
    
    /**
     * Silhouettes image to be processed.
     */
    QImage image;
    
    QString imagePath;

    float *gauss;

    bool loaded;

    int imageWidth;

    int imageHeight;

    int ker_size;
    
    /**
     * Horizon line vector
     * vector length is image width, consists of y values of the horizon line.
     * Y value has 0 in the top of the image, i.e. (0,0) is top-left corner.
     */
    double *horizonLine;

    /**
     * Extracts the horizonLine from img.
     */
    void extractHorizonLine();


    /**
     * Debug methods
     */
    void printHorizonLine(double *arr);

    void saveHorizonLineToImage(QString name);
    
    /**
     * Calculates the direction in degrees of the center of the contourlett
     */
    int calculateDirection(float fov, int i, int pause, int nSamples);

};

#endif

