/**
 *  ContourlettExtractor.cpp
 *
 *  Author: Jan Brejcha <ibrejcha@fit.vutbr.cz>, <brejchaja@gmail.com>
 *  Copyright (C) 2014  Jan Brejcha
 *
 *  OPEN SOURCE LICENCE VUT V BRNĚ
 *  Verze 1.
 *  Copyright (c) 2010, Vysoké učení technické v Brně, Antonínská 548/1, PSČ 601 90
 *  -------------------------------------------------------------------------------
 */

#include "ContourlettExtractor.h"
#include <QDebug>

ContourlettExtractor ContourlettExtractor::create(QString imgPath)
{
    QFileInfo imgInfo(imgPath);
    QString ext = imgInfo.completeSuffix();
    if (ext == "bin")
    {
        return loadFromBin(imgPath);
    }
    else return ContourlettExtractor(imgPath);
}

ContourlettExtractor::ContourlettExtractor(QString imgPath)
{
	this->imagePath = imgPath;
    this->image = QImage(imgPath);
    imageWidth = image.width();
    imageHeight = image.height();
	horizonLine = new double[image.width()];
    loaded = false;
    gauss = NULL;
	//prepare the horizon line for dividing into contourletts
}

ContourlettExtractor::ContourlettExtractor(QImage img)
{
    this->image = img;
    imageWidth = image.width();
    imageHeight = image.height();
    horizonLine = new double[image.width()];
    loaded = false;
    gauss = NULL;
}


ContourlettExtractor::ContourlettExtractor(int imageWidth, int imageHeight,
                                           double *horizonLine)
{
    this->horizonLine = horizonLine;
    loaded = true;
    this->imageWidth = imageWidth;
    this->imageHeight = imageHeight;
    gauss = NULL;
}

ContourlettExtractor::ContourlettExtractor(const ContourlettExtractor &other)
{
    this->imagePath = other.imagePath;
    this->imageWidth = other.imageWidth;
    this->imageHeight = other.imageHeight;

    int imsize = other.image.width() * other.image.height();
    if (imsize > 0)
    {
        this->image = other.image.copy();
    }
    this->horizonLine = new double[imageWidth];
    std::copy(other.horizonLine,
              other.horizonLine + imageWidth,
              this->horizonLine);
    this->ker_size = other.ker_size;
    this->gauss = new float[other.ker_size];
    std::copy(other.gauss,
              other.gauss + other.ker_size,
              this->gauss);
    this->loaded = other.loaded;
}

ContourlettExtractor::~ContourlettExtractor()
{
    if (horizonLine != NULL)
    {
        delete [] horizonLine;
        horizonLine = NULL;
    }
    if (gauss != NULL)
    {
        delete gauss;
        gauss = NULL;
    }
}

void ContourlettExtractor::saveToBin(QString filename)
{
    if (!loaded)
    {
        extractHorizonLine();
    }
    QFile file(filename);
    file.open(QIODevice::WriteOnly);
    QDataStream out(&file);
    //out.setVersion(QDataStream::Qt_4_8);
    out.setByteOrder(QDataStream::LittleEndian);
    out << image.width();
    out << image.height();
    for (int i = 0; i < image.width(); ++i)
    {
        out << (int)horizonLine[i];
    }
    file.close();
}

ContourlettExtractor ContourlettExtractor::loadFromBin(QString filename)
{
    QFile file(filename);
    file.open(QIODevice::ReadOnly);
    QDataStream in(&file);
    //in.setVersion(QDataStream::Qt_4_8);
    in.setByteOrder(QDataStream::LittleEndian);
    int imageWidth;
    int imageHeight;
    in >> imageWidth;
    in >> imageHeight;
    //qDebug() << "loaded image width: " << imageWidth << ", height: " << imageHeight;
    if (imageWidth <= 0)
    {
        throw std::runtime_error("Width of loaded horizon is not strictly positive.");
    }
    double *horizonLine = new double[imageWidth];
    int val;
    for (int i = 0; i < imageWidth; ++i)
    {
        in >> val;
        horizonLine[i] = (double)val;
    }
    file.close();

    return ContourlettExtractor(imageWidth, imageHeight, horizonLine);
}

void ContourlettExtractor::extract(float fov, float cWidth, std::vector< std::shared_ptr<LocalFeature> > &res)
{
    float pano_width_px = (360 / fov) * imageWidth;
    float cWidthPx = (cWidth / 360) * pano_width_px;
    //float sigma = cWidthPx / (2 * CONTOURLETT_SAMPLES); //baatz paper
    //float sigma = cWidthPx / (10*CONTOURLETT_SAMPLES); //baatz paper
    float sigma = cWidth / CONTOURLETT_SAMPLES; //our best experiment
    //ker_size = 6 * ((int)round(sigma)); // three sigma * 2 - baatz paper?
    ker_size = 20; // our best experiment
    gauss = new float[ker_size];

    GaussianSampler::gaussian1D(0.0, sigma, ker_size, gauss);

    this->extractHorizonLine();
    //this->printHorizonLine(horizonLine);
    //if (fov < 360)
    //{
        //run low-pass on query images only
        this->smoothenHorizonLine();
    //}

    //WARNING, takes long time! Just for debug.
    //saveHorizonLineToImage(QFileInfo(imagePath).fileName() + "_" + QString::number(fov) +"_smooth.png");

    //printHorizonLine(horizonLine);

	//extract features from horizon line
    
    int nSamples = (int)floor((fov / cWidth) * CONTOURLETT_SAMPLES);
    //printf("nsamples: %d, fov: %f\n", nSamples, fov);
    int pause = (int)floor(imageWidth/(float)nSamples);

    int nSamples_q = nSamples;
    if (pause >= 10 && fov < 360)
    {
        nSamples_q = 1 * nSamples;
    }
    int pause_q = (int)floor(imageWidth/(float)nSamples_q);

    //sliding window
    for (int i = 0; i <= nSamples_q - (CONTOURLETT_SAMPLES - 1); ++i)
    {
        int start = i * pause_q;
        int alpha = calculateDirection(fov, i, pause_q, nSamples_q);
    	//printf("direction: %d, pause: %d\n", alpha, pause);
    	//create contourlett

        //printf("image width: %d", imageWidth);
    	//printf("i: %d\n", i);

    	Contourlett *c = new Contourlett(horizonLine, pause, start, alpha);
    	if (c)
    		res.push_back(std::shared_ptr<Contourlett>(c));

        //qDebug() << "fea: " << cWidth << ", " << c->featureId();
        /*QString fea_name = pause < 10 ? "hloc_fea_quantiz_2.txt" : "hloc_fea_quantiz_10.txt";
        QFile fea_file(fea_name);
        if (fea_file.open(QIODevice::WriteOnly | QIODevice::Append))
        {
            QTextStream fea_out(&fea_file);
            fea_out << c->featureId() << " ";
            fea_file.close();
        }*/
        //std::cout << c->featureId() << " ";
    }
}




/**
 * @brief Returns only one panorama horizon line
 * @return
 */
std::vector< std::pair<int, int> > ContourlettExtractor::getSingleHorizonLinePanoramaPoints()
{
    this->extractHorizonLine();
    std::vector< std::pair<int, int>> res;
    int w = imageWidth;
    for (int i = 0; i < w; ++i)
    {
        std::pair<int, int> p(i, (int)horizonLine[i]);
        res.push_back(p);
    }
    return res;
}

/**
 * @brief Returns three panorama horizon lines next to each other
 * @return
 */
std::vector< std::pair<int, int> > ContourlettExtractor::getHorizonLinePanoramaPoints()
{
    this->extractHorizonLine();

    float pano_width_px = imageWidth;
    float cWidthPx = (10.0 / 360.0) * pano_width_px;
    float sigma = cWidthPx / (2 * CONTOURLETT_SAMPLES); //baatz paper

    //float sigma = 10; //our best experiment
    ker_size = 6 * ((int)round(sigma)); // three sigma * 2 - baatz paper?
    //ker_size = 20;
    gauss = new float[ker_size];

    GaussianSampler::gaussian1D(0.0, sigma, ker_size, gauss);

    this->smoothenHorizonLine();


	std::vector< std::pair<int, int>> res;
    int w = imageWidth;
	for (int i = 0; i < 3 * w; ++i)
	{
		std::pair<int, int> p(i, (int)horizonLine[i % w]);
		res.push_back(p);
	}
	return res;
}

std::vector< std::pair<int, int> > ContourlettExtractor::getHorizonLinePoints(float fov)
{
    this->extractHorizonLine();

    float pano_width_px = (360.0 / fov) * imageWidth;
    float cWidthPx = (10.0 / 360.0) * pano_width_px;
    float sigma = cWidthPx / (2 * CONTOURLETT_SAMPLES); //baatz paper

    //float sigma = 10; //our best experiment
    ker_size = 6 * ((int)round(sigma)); // three sigma * 2 - baatz paper?
    //ker_size = 20;
    gauss = new float[ker_size];

    GaussianSampler::gaussian1D(0.0, sigma, ker_size, gauss);

    this->smoothenHorizonLine();

	std::vector< std::pair<int, int>> res;
    int w = imageWidth;
    int h_2 = imageHeight / 2;
    int y_offset = - h_2;
	for (int i = 0; i < w; ++i)
	{
		int y = (int)horizonLine[i] + y_offset;
		std::pair<int, int> p(i, y);
		res.push_back(p);
	}
	return res;
}
int ContourlettExtractor::calculateDirection(float fov, int i, int pause, int nSamples)
{
	//calculate alpha
    int width = imageWidth;
	int width_4 = width / 4;

	if (fov == 360.0)
	{
		int start = i * pause;
		int x = start + ((CONTOURLETT_SAMPLES / 2) * pause);
		return (int)floor((fov/(float)width) * ((x - width_4 + width) % width));
	}
	//else
	//int center = (i * pause) + ((CONTOURLETT_SAMPLES / 2) * pause);
	return (int)round((fov / nSamples)*i);
}

void ContourlettExtractor::printHorizonLine(double *arr)
{
    for (int i = 0; i < imageWidth; ++i)
    {
        printf("%f ", arr[i]);
    }
    printf("\n");
}

void ContourlettExtractor::saveHorizonLineToImage(QString name)
{
    //duplicate image
    QImage out(imageWidth, imageHeight, QImage::Format_RGB32);
	//write red pixels where horizon line is.

	int index;
	int y;
    QColor red = QColor(255, 0, 0);
	for (int x = 0; x < out.width(); ++x)
	{
		y = (int)floor(horizonLine[x]);
        out.setPixelColor(x, y, red);
	}
	//safe the output image
	out.save(name);
}


void ContourlettExtractor::extractHorizonLine()
{
    if (!loaded)
    {
        int w = imageWidth;
        int h = imageHeight;
        bool horizonFound = false;
        QColor val;
        for (int x = 0; x < w; ++x)
        {
            horizonFound = false;
            for (int y = 0; y < h; ++y)
            {
                val = image.pixelColor(x, y);
                if (val.red() != 0 || val.green() != 0 || val.blue() != 0) //maybe ==0 and &
                {
                    horizonFound = true;
                    horizonLine[x] = h - (double)y;
                    break;
                }
            }
            if (!horizonFound)
                horizonLine[x] = 0;
        }

    }
    //float sigma = GAUSSIAN_KERNEL_SIZE / 2.0;
    //GaussianSampler::gaussian1D(0.0, sigma, GAUSSIAN_KERNEL_SIZE, gauss);
    //this->smoothenHorizonLine();
}

void ContourlettExtractor::smoothenHorizonLine()
{
	Imagef signal;
    int w = imageWidth;
	signal.data = horizonLine;
	signal.width = w;
	signal.height = 1;
	double *result = new double[w];
    Convolution::convolve1D(signal, result, ker_size, gauss, DIR_HORIZ);
    //printHorizonLine(result);
	//source horizon line not needed any more, delete
	//delete horizonLine;
	//we have new horizon line
    std::copy(result, result + w, horizonLine);
	//horizonLine = result;
	delete result;
};

int ContourlettExtractor::width()
{
    return imageWidth;
}

int ContourlettExtractor::height()
{
    return imageHeight;
}
