/**
 *  Author: Martin Čadík <cadik@fit.vutbr.cz>
 *          Lionel Baboud
 *          Jan Brejcha <ibrejcha@fit.vutbr.cz>, <brejchaja@gmail.com>
 *
 *  OPEN SOURCE LICENCE VUT V BRNĚ
 *  Verze 1.
 *  Copyright (c) 2010, Vysoké učení technické v Brně, Antonínská 548/1, PSČ 601 90
 *  -------------------------------------------------------------------------------
 */

#include "PlaceSet.h"

using namespace Eigen;
void Place::setCoordsSpherical(Vector3f pS) {
    longitude = 180.0 * pS.x() / M_PI;
    latitude  = 180.0 * pS.y() / M_PI;
    elevation = pS.z();
    }

Vector3f Place::getCoordsSpherical() const {
    return Vector3f(
        M_PI * longitude / 180.0,
        M_PI * latitude  / 180.0,
        elevation
        );
    }

namespace {
    bool greaterElevation(const Place &a, const Place &b) { return a.elevation > b.elevation; }    // for sorting by decreasing altitude
    }


void Place::sortDecAltitude(QList<Place> &list) {
    qSort(list.begin(), list.end(), greaterElevation);
    }
   
/************************************************************************************************************/

const QDir PlaceSet::root = QDir("data/places");

PlaceSet::PlaceSet(QString fileName) : fileName(fileName) {}

/*PlaceSet PlaceSet::load()
{
    QString placeName = root.readingFileName(QString::null, QString(".xml"));

    return PlaceSet(placeName);
}*/

void PlaceSet::add(Place p) {
    write(p);
    }

Place PlaceSet::get(QString shortcut) const {
    QList<Place> set = read();
    int i = set.indexOf(Place(shortcut));
    return (i >= 0) ? set[i] : Place();
    }

QList<Place> PlaceSet::getAll() const {
    return readEfficient();
    }

QList<Place> PlaceSet::readEfficient() const
{
    QList<Place> res;

    QString path = root.filePath(fileName);
    QFile places_file(path);
    if (!places_file.open(QIODevice::ReadOnly))
    {
        std::cerr << "Cannot open places file: " << path.toStdString() << std::endl;
        return res;
    }

    QXmlStreamReader xml;
    xml.setDevice(&places_file);

    while(!xml.atEnd() && !xml.hasError())
    {
        xml.readNext();
        if (xml.isStartElement() && !xml.hasError() && xml.name() != NULL)
        {
            if (xml.name().toString() == "Place")
            {
                Place p;
                QXmlStreamAttributes attrs = xml.attributes();
                p.shortcut = attrs.value("shortcut").toString();
                p.name = attrs.value("name").toString();
                p.longitude = attrs.value("longitude").toString().toFloat();
                p.latitude = attrs.value("latitude").toString().toFloat();
                p.elevation = attrs.value("elevation").toString().toFloat();
                p.heatmap_val = attrs.value("heatmap_val").toString().toFloat();
                p.total_edge_len = attrs.value("total_edge_len").toString().toFloat();
                res << p;
                //qDebug() << p.shortcut;
            }
        }
    }
    if (xml.hasError()) {
        std::cerr << "Unable to load places for reason: " << xml.error() << std::endl;
        return res;
    }

    places_file.close();
    return res;
}


double PlaceSet::getXMLDoubleAttrib(  QDomElement element,
                                      QString attrib_name,
                                      double defaultValue) const
{
    bool ok;
    double res = element.attribute(attrib_name).toDouble(&ok);
    if (ok)
    {
        return res;
    }
    return defaultValue;

}

QList<Place> PlaceSet::read() const {
    QList<Place> res;
    QDomDocument doc = XMLUtil::loadQDomDocFromFile(root.absoluteFilePath(fileName));
    qDebug() << "Places loaded into memory.";
    if (doc.isNull())
    {
        qDebug() << "could not read file " << fileName << " in directory " << root.dirName();
        return res;
    }

    // lecture des attributs :
    QDomElement eGlobal = doc.documentElement();
    QDomElement child = eGlobal.firstChild().toElement();
    while (!child.isNull()) {
        if (child.tagName() == "Place") {
            Place place;
            place.shortcut  = child.toElement().attribute("shortcut");
            place.name      = child.toElement().attribute("name");
            place.longitude = getXMLDoubleAttrib(child, "longitude", 0.0);
            place.latitude  = getXMLDoubleAttrib(child, "latitude",  0.0);
            place.elevation = getXMLDoubleAttrib(child, "elevation", 0.0);
            place.heatmap_val = getXMLDoubleAttrib(child, "heatmap_val", -1.0);
            place.total_edge_len = getXMLDoubleAttrib(child, "tot_edge_len", 0.0);
            res << place;
            }
        else
            qDebug() << "unsupported tag for PlaceSet : " << child.tagName();
        
        child = child.nextSibling().toElement();
        }
    
    return res;
    }

void PlaceSet::writeAll(QList<Place> places)
{
	write(places);
}

void PlaceSet::saveXML(QDomDocument doc, QString fileName) const
{
    QFile file(fileName);
    if(!file.open( QIODevice::WriteOnly | QIODevice::Text))
    {
        throw std::runtime_error("Failed to open file "
                                 + fileName.toStdString() + " for writing.");

    }
    QTextStream stream(&file);
    stream << doc.toString();
    file.close();
}

void PlaceSet::write(Place p) { write(QList<Place>() << p); }
void PlaceSet::write(QList<Place> set) {
    QDomDocument doc("PlaceSet");
    QDomElement eGlobal = doc.createElement("TerrainTiling");

    //@@ rather use a QSet..

    if (root.entryList().contains(fileName)) {
        QList<Place> oldSet = read();
        foreach (const Place &p, oldSet) if (!set.contains(p)) set.push_front(p);
        }
    
    foreach (const Place &place, set) {
        QDomElement ePlace = doc.createElement("Place");
        ePlace.setAttribute("shortcut",     place.shortcut);
        ePlace.setAttribute("name",         place.name);
        ePlace.setAttribute("longitude",    QString::number(place.longitude ));
        ePlace.setAttribute("latitude",     QString::number(place.latitude  ));
        ePlace.setAttribute("elevation",    QString::number(place.elevation ));
        //ePlace.setAttribute("heatmap_val", QString::number(place.heatmap_val));
        //ePlace.setAttribute("tot_edge_len", QString::number(place.total_edge_len));
        eGlobal.appendChild(ePlace);
    }
    doc.appendChild(eGlobal);
    saveXML(doc, root.absoluteFilePath(fileName));
}

