/*
 * The_xmloperator_project Software License, Version 1.7
 *
 * Copyright (c) 2000 - 2003 The_xmloperator_project.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *      "This product includes or uses software developped
 *       by The_xmloperator_project (http://www.xmloperator.net/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. Products derived from this software may not be called "xmloperator",
 *    nor may "xmloperator" appear in their name, without prior written
 *    permission. For written permission, please contact
 *    the xmloperator project administrator.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE_XMLOPERATOR_PROJECT OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * Further information can be found on the project's site
 * (http://www.xmloperator.net/).
 */
package xmltorng.i2s.util;

import java.util.Arrays;

/**
 * This object represents a matix of <code>n</code> x <code>n</code>
 *   integer values.
 */
public final class NNIntMatrix {
  private int[][] matrix;
  private int n;

  /**
   * Default constructor. Set <code>n</code> to 0.
   */
  public NNIntMatrix() {
    this(4);
    this.n = 0;
  }

  /**
   * Constructor with dimension argument.
   */
  public NNIntMatrix(int n) {
    if (n < 1)
      throw new IllegalArgumentException();
    this.matrix = new int[n][n];
    this.n = n;
    for (int k = 0; k < n; k++)
      Arrays.fill(matrix[k], 0);
  }

  public int getDimension() {
    return this.n;
  }

  public void setDimension(int n) {
    if (n < 1)
      throw new IllegalArgumentException();
    if (n > this.matrix.length) {
      if (n > 0x1000)
        throw new ArithmeticException("Too large matrix");
      int size = this.matrix.length;
      int new_size = 2 * size;
      while (n > new_size)
        new_size = 2 * new_size;
      int[][] new_matrix = new int[new_size][new_size];
      for (int k = 0; k < new_size; k++)
        Arrays.fill(new_matrix[k], 0);
      for (int k = 0; k < size; k++)
        System.arraycopy(this.matrix[k], 0, new_matrix[k], 0, size);
      this.matrix = new_matrix;
    }
    this.n = n;
  }

  public int get(int index1, int index2) {
    if (index1 >= this.n || index2 >= this.n)
      throw new IllegalArgumentException();
    return matrix[index1][index2];
  }

  public void set(int index1, int index2, int value) {
    if (index1 >= this.n || index2 >= this.n)
      throw new IllegalArgumentException();
    this.matrix[index1][index2] = value;
  }

  public void setSymetricaly(int index1, int index2, int value) {
    if (index1 >= this.n || index2 >= this.n)
      throw new IllegalArgumentException();
    this.matrix[index1][index2] = value;
    this.matrix[index2][index1] = value;
  }

  public void setAntisymetricaly(int index1, int index2, int value) {
    if (index1 >= this.n || index2 >= this.n
        || (index1 == index2 && value != 0))
      throw new IllegalArgumentException();
    this.matrix[index1][index2] = value;
    this.matrix[index2][index1] = - value;
  }

  /**
   * Considering a symetrical or anti-symetrical matrix
   *   that represents associations between n elements,
   *   this method collects the n elements by groups.
   * Two elements associated by a not null value belong to the same group.
   * Elements of a group take the same number.
   * Groups are numbered from 1.
   * The group index of each element is returned in <code>groupIndexes</code>.
   * Return the group count.
   */
  public int collectConnectedElements(int[] groupIndexes) {
    if (this.n == 0)
      return 0;
    if (groupIndexes.length < this.n)
      throw new IllegalArgumentException();
    Arrays.fill(groupIndexes, 0, this.n - 1, 0);
    int groupCount = 0;
    while (true) {
      int elementIndex = 0;
      while (elementIndex < this.n && groupIndexes[elementIndex] != 0)
        elementIndex++;
      if (elementIndex >= this.n)
        break;
      groupIndexes[elementIndex] = ++groupCount;
      setGroupIndexToConnectedElements(groupIndexes, elementIndex, groupCount);
    }
    return groupCount;
  }

  private final void setGroupIndexToConnectedElements(
      int[] groupIndexes, int elementIndex, int groupIndex) {
    for (int k = 0; k < this.n; k++)
      if (this.matrix[elementIndex][k] != 0 && groupIndexes[k] == 0) {
        groupIndexes[k] = groupIndex;
        setGroupIndexToConnectedElements(groupIndexes, k, groupIndex);
      }
  }
}
