package convert.relaxng.output.rng;

import convert.relaxng.edit.AbstractVisitor;
import convert.relaxng.edit.Annotated;
import convert.relaxng.edit.AnnotationChild;
import convert.relaxng.edit.AnyNameNameClass;
import convert.relaxng.edit.AttributeAnnotation;
import convert.relaxng.edit.AttributePattern;
import convert.relaxng.edit.ChoiceNameClass;
import convert.relaxng.edit.Component;
import convert.relaxng.edit.CompositePattern;
import convert.relaxng.edit.Container;
import convert.relaxng.edit.DataPattern;
import convert.relaxng.edit.DefineComponent;
import convert.relaxng.edit.DivComponent;
import convert.relaxng.edit.ElementAnnotation;
import convert.relaxng.edit.ExternalRefPattern;
import convert.relaxng.edit.GrammarPattern;
import convert.relaxng.edit.IncludeComponent;
import convert.relaxng.edit.NameClass;
import convert.relaxng.edit.NameClassedPattern;
import convert.relaxng.edit.NameNameClass;
import convert.relaxng.edit.NsNameNameClass;
import convert.relaxng.edit.Param;
import convert.relaxng.edit.Pattern;
import convert.relaxng.edit.UnaryPattern;
import convert.relaxng.edit.ValuePattern;
import convert.util.VoidValue;
import convert.relaxng.parse.Context;
import convert.xml.util.WellKnownNamespaces;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

class Analyzer extends AbstractVisitor {

  private VoidValue visitAnnotated(Annotated anno) {
    if (anno.getAttributeAnnotations().size() > 0
        || anno.getChildElementAnnotations().size() > 0
        || anno.getFollowingElementAnnotations().size() > 0)
      noteContext(anno.getContext());
    visitAnnotationAttributes(anno.getAttributeAnnotations());
    visitAnnotationChildren(anno.getChildElementAnnotations());
    visitAnnotationChildren(anno.getFollowingElementAnnotations());
    return VoidValue.VOID;
  }

  private void visitAnnotationAttributes(List<AttributeAnnotation> list) {
    for (int i = 0, len = list.size(); i < len; i++) {
      AttributeAnnotation att = list.get(i);
      if (att.getNamespaceUri().length() != 0)
        noteNs(att.getPrefix(), att.getNamespaceUri());
    }
  }

  private void visitAnnotationChildren(List<AnnotationChild> list) {
    for (int i = 0, len = list.size(); i < len; i++) {
      AnnotationChild ac = list.get(i);
      if (ac instanceof ElementAnnotation) {
        ElementAnnotation elem = (ElementAnnotation)ac;
        if (elem.getPrefix() != null)
          noteNs(elem.getPrefix(), elem.getNamespaceUri());
        visitAnnotationAttributes(elem.getAttributes());
        visitAnnotationChildren(elem.getChildren());
      }
    }
  }

  public VoidValue visitPattern(Pattern p) {
    return visitAnnotated(p);
  }

  public VoidValue visitDefine(DefineComponent c) {
    visitAnnotated(c);
    return c.getBody().accept(this);
  }

  public VoidValue visitDiv(DivComponent c) {
    visitAnnotated(c);
    return visitContainer(c);
  }

  public VoidValue visitInclude(IncludeComponent c) {
    visitAnnotated(c);
    noteInheritNs(c.getNs());
    return visitContainer(c);
  }

  public VoidValue visitGrammar(GrammarPattern p) {
    visitAnnotated(p);
    return visitContainer(p);
  }

  private VoidValue visitContainer(Container c) {
    List<Component> list = c.getComponents();
    for (int i = 0, len = list.size(); i < len; i++)
      (list.get(i)).accept(this);
    return VoidValue.VOID;
  }

  public VoidValue visitUnary(UnaryPattern p) {
    visitAnnotated(p);
    return p.getChild().accept(this);
  }

  public VoidValue visitComposite(CompositePattern p) {
    visitAnnotated(p);
    List<Pattern> list = p.getChildren();
    for (int i = 0, len = list.size(); i < len; i++)
      (list.get(i)).accept(this);
    return VoidValue.VOID;
  }

  public VoidValue visitNameClassed(NameClassedPattern p) {
    p.getNameClass().accept(this);
    return visitUnary(p);
  }

  public VoidValue visitAttribute(AttributePattern p) {
    NameClass nc = p.getNameClass();
    if (nc instanceof NameNameClass
        && ((NameNameClass)nc).getNamespaceUri().equals(""))
      return visitUnary(p);
    return visitNameClassed(p);
  }

  public VoidValue visitChoice(ChoiceNameClass nc) {
    visitAnnotated(nc);
    List<NameClass> list = nc.getChildren();
    for (int i = 0, len = list.size(); i < len; i++)
      (list.get(i)).accept(this);
    return VoidValue.VOID;
  }

  public VoidValue visitValue(ValuePattern p) {
    visitAnnotated(p);
    if (!p.getType().equals("token") || !p.getDatatypeLibrary().equals(""))
      noteDatatypeLibrary(p.getDatatypeLibrary());
    for (Map.Entry<String, String> entry : p.getPrefixMap().entrySet()) {
      noteNs(entry.getKey(), entry.getValue());
    }
    return VoidValue.VOID;
  }

  public VoidValue visitData(DataPattern p) {
    visitAnnotated(p);
    noteDatatypeLibrary(p.getDatatypeLibrary());
    Pattern except = p.getExcept();
    if (except != null)
      except.accept(this);
    for (Param param : p.getParams())
      visitAnnotated(param);
    return VoidValue.VOID;
  }

  public VoidValue visitExternalRef(ExternalRefPattern p) {
    visitAnnotated(p);
    noteInheritNs(p.getNs());
    return VoidValue.VOID;
  }

  public VoidValue visitName(NameNameClass nc) {
    visitAnnotated(nc);
    noteNs(nc.getPrefix(), nc.getNamespaceUri());
    return VoidValue.VOID;
  }

  public VoidValue visitAnyName(AnyNameNameClass nc) {
    visitAnnotated(nc);
    NameClass except = nc.getExcept();
    if (except != null)
      except.accept(this);
    return VoidValue.VOID;
  }

  public VoidValue visitNsName(NsNameNameClass nc) {
    visitAnnotated(nc);
    noteInheritNs(nc.getNs());
    NameClass except = nc.getExcept();
    if (except != null)
      except.accept(this);
    return VoidValue.VOID;
  }

  private String datatypeLibrary = null;
  private final Map<String, String> prefixMap = new HashMap<String, String>();
  private boolean haveInherit = false;
  private Context lastContext = null;
  private String noPrefixNs = null;

  private void noteDatatypeLibrary(String uri) {
    if (datatypeLibrary == null || datatypeLibrary.length() == 0)
      datatypeLibrary = uri;
  }

  private void noteInheritNs(String ns) {
    if (ns == NameClass.INHERIT_NS)
      haveInherit = true;
    else
      noPrefixNs = ns;
  }

  private void noteNs(String prefix, String ns) {
    if (ns == NameClass.INHERIT_NS) {
      haveInherit = true;
      return;
    }
    if (prefix == null)
      prefix = "";
    if (ns == null || (ns.length() == 0 && prefix.length() != 0) || prefixMap.containsKey(prefix))
      return;
    prefixMap.put(prefix, ns);
  }

  private void noteContext(Context context) {
    if (context == null || context == lastContext)
      return;
    lastContext = context;
    for (Enumeration e = context.prefixes(); e.hasMoreElements();) {
      String prefix = (String)e.nextElement();
      noteNs(prefix, context.resolveNamespacePrefix(prefix));
    }
  }

  Map<String, String> getPrefixMap() {
    if (haveInherit)
      prefixMap.remove("");
    else if (noPrefixNs != null && !prefixMap.containsKey(""))
      prefixMap.put("", noPrefixNs);
    prefixMap.put("xml", WellKnownNamespaces.XML);
    return prefixMap;
  }

  String getDatatypeLibrary() {
    return datatypeLibrary;
  }

}
