/*
 *   lrd_engine.h
 *   $Id$
 *   
 *   ----
 *   Department of Computer Graphics and Multimedia
 *   Faculty of Information Technology, Brno University of Technology
 *   
 *   ----
 */

#ifndef _LRD_ENGINE_H_
#define _LRD_ENGINE_H_

#define MEASURE_PERFORMANCE 0 ///< Gather statistics about evaluation process
#define WALDBOOST_EVAL      1 ///< Use WaldBoost threshold (early termination)


#include <cv.h>
#include <cxcore.h>
#include <vector>
#include "imagetools.h"


#if MEASURE_PERFORMANCE
// performance counters
extern unsigned long long hypotheses;
extern unsigned long long windows;
extern unsigned long long totalWindows;
extern unsigned long long totalHypotheses;
extern unsigned long long scanTSC;
extern unsigned long long evalTSC;
#endif


/// A detection structure.
/// Holds the response and the position in an image.
struct TDetection
{
    float response;
    float angle;
    CvRect area;
};

typedef std::vector<TDetection> TDetectionList;


/// A single weak hypothesis.
/// 3x3 LRD feature, WaldBoost threshold and ptr to lookup table.
struct TStage
{
	// Feature and hypothesis parameters
    int x, y, w, h;
    char A, B; // feature parameters
	float theta_b; // wald_boost threshold
	float * alpha; // Ptr to table with alphas
    
    char szType;  // 2 * (h-1) + (w-1)
    char posType; // (4 * (y % 4) + (x % 4))
    //int rank[8];  // rank[2 * T + 0]; - index of rank pixel A for sample shift type T (+1 for B)
};


/// Classifier structure.
/// Holds informations about a classifier - parameters,
/// weak hypotheses, alphas and convolutions.
struct TClassifier
{
    unsigned stageCount; ///< Number of stages
    unsigned alphaCount; ///< Number of alphas per stage

    float threshold; ///< Final classification threshold
    
    unsigned width, height; ///< Size of scanning window
    
    TStage * stage; ///< List of stages
    float * alpha; ///< List of alphas
};

///
/// Convolution management
///

std::vector<TConvolution*> * createConvolutions(CvSize sz);
void releaseConvolutions(std::vector<TConvolution*> ** conv);
int * prepareColAddressTable(std::vector<TConvolution*> & c);
int * prepareRowAddressTable(std::vector<TConvolution*> & c);

///
/// LRD Engine
///

int initClassifier(TClassifier * classifier);
//void updateClassifierParams(TClassifier * classifier, std::vector<TConvolution*> & conv);

typedef int Rank[8];
void calculateClassifierRanks(Rank * ranks, const TClassifier * classifier, const std::vector<TConvolution*> & conv);
    
/// Scan given image with classifier.
unsigned scanConvolvedImage(
        const std::vector<TConvolution*> &, // The convolved image
        const int * xtable, const int * ytable,   // Tables for addressing the convolutions
        const TClassifier*, const Rank*,                 // The classifier
        TDetectionList::iterator first,  // Detection vector
        TDetectionList::iterator last    // The end of det. vector
        );
/*
unsigned scanIntensityImage(
        IplImage * image,
        TClassifier*,                 // The classifier
        TDetectionList::iterator first,  // Detection vector
        TDetectionList::iterator last    // The end of det. vector
        );
        */

#endif

